import * as React from 'react';
import { match, Redirect } from 'react-router-dom';
import { APIError } from 'lib/api';
// @ts-ignore
import { Button, ButtonType, CoreText, Display, FlexDirection, FormGroupOrientation, JustifyContent, Layout, TextAlign, TextType } from 'twitch-core-ui';
import { getQueryResult } from '../../api';
import { QueryResult } from '../../model';
import * as Reports from '../../reports';

interface URLParams {
  report_id: string;
  region: string;
  content_id: string;
  session_id: string;
  master_session_id: string;
}

interface Props {
  match: match<URLParams>;
}

export interface State {
  navigate: string | undefined;
  loading: boolean;
  result: QueryResult | undefined;
  error?: string;
}

export class ReportDetail extends React.Component<Props, State> {
  public state: State = {
    navigate: undefined,
    loading: false,
    result: undefined,
  };

  private retryTimeout: NodeJS.Timer;

  public render() {
    if (this.state.navigate !== undefined) {
      return <Redirect to={this.state.navigate} push={true} />;
    }

    if (this.state.error) {
      return (
        <Layout textAlign={TextAlign.Center}>
          <CoreText type={TextType.H3}>{this.state.error}</CoreText>
          <CoreText>I promise I will make this page look less bad soon</CoreText>
        </Layout>
      );
    }

    if (this.state.result) {
      // Only one of these will be populated in the response
      if (this.state.result.PlaySessionInfo) {
        return <Reports.PlaySessionInfo.Page data={this.state.result.PlaySessionInfo} />;
      }

      if (this.state.result.BroadcastInfo) {
        return <Reports.BroadcastSessionInfo.Page data={this.state.result.BroadcastInfo} />;
      }
    }

    return (
      <Layout>
        Loading Query
      </Layout>
    );
  }

  public componentDidMount() {
    this.getQueryResult();
  }

  public componentWillUnmount() {
    if (this.retryTimeout) {
      clearTimeout(this.retryTimeout);
    }
  }

  public componentDidUpdate(prevProps: Props) {
    if (this.props.match.params.report_id === prevProps.match.params.report_id) {
      return;
    }

    this.setState({
      loading: true,
    });

    this.getQueryResult();
  }

  private getQueryResult = () => {
    getQueryResult(this.props.match.params.report_id).then((result) => {
      this.setState({
        result,
        error: undefined,
      });
    }, (reason: APIError) => {
      let errorMsg = 'This query has failed, please contact us if this happens persistently';
      switch (reason.code) {
        case 'AWAITING':
          this.retryTimeout = setTimeout(this.getQueryResult, 10000);
          errorMsg = 'This query is still running, you can just leave this page right here and we will load the results when they are ready.';
          break;
      }
      this.setState({
        error: errorMsg,
      });
    });
  }
}
