import * as React from 'react';
import { Redirect } from 'react-router-dom';
import { TimeStamp } from 'lib/api';
import { APIError } from 'lib/api';
import { submitQuery } from 'modules/reports/api';
import { Result } from 'modules/reports/model/broadcast-session-info';
import { Button, CoreText, Display, FlexDirection, Layout, Table, TableBody, TableCell, TableHeader, TableHeading, TableRow, TextAlign, TextType } from 'twitch-core-ui';
//import './styles.scss';

export interface Props {
  data: Result;
}

export interface State {
  navigate?: string;
}

export class Page extends React.Component<Props, State> {
  public state: State = {
    navigate: undefined,
  };

  public render() {
    if (this.state.navigate !== undefined) {
      return <Redirect to={this.state.navigate} push={true} />;
    }

    if (!this.props.data.play_sessions_by_minutes_watched && !this.props.data.play_sessions_by_bemw) {
      return (
        <Layout textAlign={TextAlign.Center}>
          <CoreText type={TextType.H3}>Empty Result</CoreText>
        </Layout>
      );
    }

    let playSessionsByMw = (<></>);

    if (this.props.data.play_sessions_by_minutes_watched) {
      playSessionsByMw = (
        <Layout padding={3}>
          <Layout padding={{ bottom: 2 }}>
            <CoreText type={TextType.H3}>Top Play Sessions by Minutes Watched</CoreText>
          </Layout>
          <Table>
            <TableHeader>
              <TableHeading label="Minutes Watched" />
              <TableHeading label="Play Session ID" />
              <TableHeading label="Time" />
              <TableHeading label="Client Time" />
              <TableHeading label="Last Minute Watched Time" />
              <TableHeading label="Broadcast ID" />
              <TableHeading label="Manifest Cluster" />
              <TableHeading label="Node" />
              <TableHeading label="ASN" />
              <TableHeading label="" />
            </TableHeader>
            <TableBody>
              {this.props.data.play_sessions_by_minutes_watched.map((playSession, index) => (
                <TableRow key={index}>
                  <TableCell>{playSession.minutes_watched}</TableCell>
                  <TableCell>{playSession.play_session_summary.play_session_id}</TableCell>
                  <TableCell>{this.context.time.defaultFormat(playSession.play_session_summary.time)}</TableCell>
                  <TableCell>{this.context.time.defaultFormat(playSession.play_session_summary.client_time)}</TableCell>
                  <TableCell>{this.context.time.defaultFormat(playSession.play_session_summary.last_minute_watched_time)}</TableCell>
                  <TableCell>{playSession.play_session_summary.broadcast_id}</TableCell>
                  <TableCell>{playSession.play_session_summary.manifest_cluster}</TableCell>
                  <TableCell>{playSession.play_session_summary.node}</TableCell>
                  <TableCell>{playSession.play_session_summary.asn}</TableCell>
                  <TableCell><Button onClick={() => this.generatePlaySessionReport(playSession.play_session_summary.play_session_id, playSession.play_session_summary.time, playSession.play_session_summary.last_minute_watched_time)}>Generate Play Session Report</Button></TableCell>
                </TableRow>
              ))}
            </TableBody>
          </Table>
        </Layout>
      );
    }

    let playSessionsByBeMw = (<></>);

    if (this.props.data.play_sessions_by_bemw) {
      playSessionsByBeMw = (
        <Layout padding={3}>
          <Layout padding={{ bottom: 2 }}>
            <CoreText type={TextType.H3}>Top Play Sessions by Buffer Empties per Minutes Watched</CoreText>
          </Layout>
          <Table>
            <TableHeader>
              <TableHeading label="Minutes Watched" />
              <TableHeading label="Buffer Empties" />
              <TableHeading label="Buffer Empties per Minute Watched" />
              <TableHeading label="Play Session ID" />
              <TableHeading label="Time" />
              <TableHeading label="Client Time" />
              <TableHeading label="Last Minute Watched Time" />
              <TableHeading label="Broadcast ID" />
              <TableHeading label="Manifest Cluster" />
              <TableHeading label="Node" />
              <TableHeading label="ASN" />
              <TableHeading label="" />
            </TableHeader>
            <TableBody>
              {this.props.data.play_sessions_by_bemw.map((playSession, index) => (
                <TableRow key={index}>
                  <TableCell>{playSession.minutes_watched}</TableCell>
                  <TableCell>{playSession.buffer_empties}</TableCell>
                  <TableCell>{playSession.bemw}</TableCell>
                  <TableCell>{playSession.play_session_summary.play_session_id}</TableCell>
                  <TableCell>{this.context.time.defaultFormat(playSession.play_session_summary.time)}</TableCell>
                  <TableCell>{this.context.time.defaultFormat(playSession.play_session_summary.client_time)}</TableCell>
                  <TableCell>{this.context.time.defaultFormat(playSession.play_session_summary.last_minute_watched_time)}</TableCell>
                  <TableCell>{playSession.play_session_summary.broadcast_id}</TableCell>
                  <TableCell>{playSession.play_session_summary.manifest_cluster}</TableCell>
                  <TableCell>{playSession.play_session_summary.node}</TableCell>
                  <TableCell>{playSession.play_session_summary.asn}</TableCell>
                  <TableCell><Button onClick={() => this.generatePlaySessionReport(playSession.play_session_summary.play_session_id, playSession.play_session_summary.time, playSession.play_session_summary.last_minute_watched_time)}>Generate Play Session Report</Button></TableCell>
                </TableRow>
              ))}
            </TableBody>
          </Table>
        </Layout>
      );
    }

    return (
      <Layout display={Display.Flex} flexDirection={FlexDirection.Column}>
        {playSessionsByMw}
        {playSessionsByBeMw}
      </Layout>
    );
  }

  private generatePlaySessionReport = (playSessionID: string, start: TimeStamp, end: TimeStamp) => {
    // We're going to pad the start and end times by 5 mins to ensure we don't miss any events in the query
    let padding = 300000; // 5 minutes

    let sessionStart = new Date((start.seconds * 1000) - padding);
    let sessionEnd = new Date((end.seconds * 1000) + padding);

    submitQuery({
      query_type: 'play_session_info',
      start: sessionStart,
      end: sessionEnd,
      play_session_id: playSessionID,
    }).then((queryID: string) => {
      this.setState({
        navigate: '/reports/' + queryID,
      });
    }, (reason: APIError) => {
      this.context.showModal(
        (
          <div>{reason.message}</div>
        ),
        {
          title: 'Error Generating Report',
          onClose: () => {
            this.context.hideModal();
          },
        },
      );
    });
  }
}
