import { API } from 'lib/api';
import { Duration, TimeStamp } from 'lib/api';
import { SECEvent } from 'common/sec/model'

export interface GetSessionResponse {
  sessions: Session[];
}

export interface Session {
  session_id: string;
  related_sessions: Session[];
  start: TimeStamp;
  end: TimeStamp | undefined;
  ingest_edge: string;
  broadcast_format: string;
}

export interface GetSessionDetailResponse {
  session: SessionDetail;
}

export interface SessionDetail {
  session_id: string;
  ingest_session: IngestSession;
  related_sessions: Session[];
  rtmp_session: RtmpSession;
  backup_ingest_session: BackupIngestSession;
  transcode_sessions: TranscodeSession[];
  framerates: FrameRate[];
  bitrates: Bitrate[];
  starvations: Starvation[];
}

export interface IngestSession {
  stream_up: TimeStamp;
  stream_down: TimeStamp | undefined;
  ingest_host: string;
  broadcast_format: string;
  ingest_proxy: string;
  rtmp_session: RtmpSession;
}

export interface Starvation {
  start: TimeStamp;
  duration: Duration;
}

export interface IPInfo {
  country: string;
  city: string;
  asn: number;
  isp_name: string;
}

export interface RtmpSession {
  audio_codecs: string;
  video_codecs: string;
  video_resolution_height: number;
  video_resolution_width: number;
  avc_level: number;
  idr_interval: number;
  segment_duration: number;
  rtmp_metadata: string;
  rtmp_flags: string;
  rtmp_exit_reason: string;
  encoder: string;
  client_ip: string;
  ip_info: IPInfo;
  video_data_rate: number;
  video_frame_rate: number;
  audio_sample_rate: number;
  audio_data_rate: number;
  audio_channels: number;
  audio_is_stereo: boolean;
  audio_sample_size: number;
}

export interface BackupSession {
  session_id: string;
  broadcast_format: string;
  stitched_timestamp: TimeStamp;
}

export interface BackupIngestSession {
  stitched_from: BackupSession[];
  stitched_to: BackupSession[];
}

export interface TranscodeSession {
  host: string;
  profile: string;
  stack_version: number;
  ingest_origin: string;
  audio_codecs: string;
  video_codecs: string;
  timestamp: TimeStamp;
  recording_s3_bucket: string;
  recording_prefix: string;
}

export interface FrameRate {
  timestamp: TimeStamp;
  value_fps: number;
}

export interface Bitrate {
  timestamp: TimeStamp;
  value_kbps: number;
}

export interface GetSECEventsResponse {
  stream_events: SECEvent[];
}

export async function getSessions(customerID: string, region: string, contentID: string, before: string): Promise<Session[]> {
  let path = `/api/sessionviewer/` + customerID + '/' + region + '/' + contentID;

  if (before) {
    path += '?before=' + encodeURIComponent(before)
  }

  const response = await API.get<GetSessionResponse>(path, {});

  if (response.error) {
    return Promise.reject(response.error);
  }

  if (response.body === undefined) {
    return [];
  }
  return response.body.sessions;
}

export async function getSessionDetail(customerID: string, region: string, contentID: string, sessionID: string): Promise<SessionDetail | undefined> {
  const path = `/api/sessionviewer/` + customerID + '/' + region + '/' + contentID + '/' + sessionID;

  const response = await API.get<GetSessionDetailResponse>(path, {});

  if (response.error) {
    return Promise.reject(response.error);
  }

  if (response.body === undefined) {
    return undefined;
  }
  return response.body.session;
}

export async function getSECEvents(customerID: string, region: string, contentID: string, sessionID: string): Promise<SECEvent[] | undefined> {
  const path = `/api/sessionviewer/` + customerID + '/' + region + '/' + contentID + '/' + sessionID + '/events';

  const response = await API.get<GetSECEventsResponse>(path, {});

  if (response.error) {
    return Promise.reject(response.error);
  }

  if (response.body === undefined) {
    return undefined;
  }
  return response.body.stream_events;
}
