import 'dygraphs/dist/dygraph.css';
import Dygraph from 'dygraphs/index.es5';
import * as React from 'react';
import { Layout } from 'twitch-core-ui';
import { BackupSession } from '../../api';

interface Props {
  datasetLabel: string;
  bitrateData: { x: Date, y: number }[];
  framerateData: { x: Date, y: number }[];
  failoverData: { x: Date, y: BackupSession, onClick: () => void }[];
  starvationData: { x: Date, y: string }[];
}

export class TimeseriesChart extends React.Component<Props> {
  public chart?: Dygraph;
  private container: HTMLElement | null;

  public render() {
    return (
      <Layout>
        <div style={{ width: '100%' }} ref={(el) => (this.container = el)} />
      </Layout>
    );
  }

  public componentDidUpdate(prevProps: Props) {
    if (!this.chart || (this.props.bitrateData === prevProps.bitrateData && this.props.framerateData === prevProps.framerateData && this.props.failoverData === prevProps.failoverData && this.props.starvationData === prevProps.starvationData)) {
      return;
    }

    let data = this.getData();
    let annotations = this.getAnnotations();
    data = this.mergeAnnotationData(data, annotations);

    if (this.chart) {
      this.chart.updateOptions({
        file: data,
      });

      this.chart.setAnnotations(annotations);
    }
  }

  public componentDidMount() {
    let data = this.getData();
    let annotations = this.getAnnotations();
    data = this.mergeAnnotationData(data, annotations);

    if (!this.container) {
      return;
    }

    this.chart = new Dygraph(this.container, data, {
      labels: ['time', 'bitrate', 'framerate', 'annotations'],
      connectSeparatedPoints: true,
      legend: 'follow',
      labelsShowZeroValues: false,
      showRangeSelector: true,
      includeZero: true,
      digitsAfterDecimal: 0,
      rollPeriod: 5,
      showRoller: true,
      labelsSeparateLines: true,
      series: {
        bitrate: {
          axis: 'y1',
          showInRangeSelector: true,
          strokeWidth: 2,
        },
        framerate: {
          axis: 'y2',
          strokeWidth: 2,
        },
        annotations: { axis: 'y2' },
      },
      axes: {
        y2: {
          independentTicks: true,
          ticker: () => {
            return [
              { v: 0, label: '0' },
              { v: 10, label: '10' },
              { v: 20, label: '20' },
              { v: 30, label: '30' },
              { v: 40, label: '40' },
              { v: 50, label: '50' },
              { v: 60, label: '60' },
              { v: 70, label: '70' },
              { v: 80, label: '70' },
              { v: 90, label: '90' },
            ];
          },
        },
      },
      interactionModel: {
        wheel: (e, g) => {
          return this.scroll(e, g);
        },
      },

    });

    this.chart.setAnnotations(annotations);
  }

  private mergeAnnotationData(data: [][], annotations: any[]): [][] {
    // We're basically going to create a separate series just for the points where we want annotations,
    // They will all be at 0, (THANKS FOR THE IDEA SAM)
    // We still have to merge it with the x axis from the source data though which is super meh....
    let currentA = 0;

    for (let i = 0; i < data.length; i++) {
      let dItem = data[i];
      for (; currentA < annotations.length; currentA++) {
        let aItem = annotations[currentA];
        // @ts-ignore
        if (dItem[0].getTime() == aItem.xVal.getTime()) {
          // @ts-ignore
          dItem[3] = 0;
          // @ts-ignore
        } else if (dItem[0].getTime() < aItem.xVal.getTime() && data[i + 1] && data[i + 1][0].getTime() > aItem.xVal.getTime()) {
          // @ts-ignore
          data.splice(++i, 0, [aItem.xVal, null, null, 0]);
        } else {
          break;
        }
      }
    }
    return data;
  }

  private getAnnotations(): any[] {
    let annotations: any[] = [];

    for (let failover of this.props.failoverData) {
      annotations.push({
        series: 'annotations',
        x: failover.x.getTime(),
        xVal: failover.x,
        width: 24,
        height: 24,
        tickHeight: 8,
        attachAtBottom: true,
        shortText: 'F',
        clickHandler: failover.onClick,
        text: 'Failed over to ' + failover.y.broadcast_format,
      });
    }

    for (let starvation of this.props.starvationData) {
      annotations.push({
        series: 'annotations',
        x: starvation.x.getTime(),
        xVal: starvation.x,
        width: 24,
        height: 24,
        tickHeight: 8,
        attachAtBottom: true,
        shortText: starvation.y == 'start' ? 'SS' : 'SE',
        text: 'starvation ' + starvation.y,
      });
    }

    return annotations.sort((a: any, b: any) => {
      if (a.x > b.x) {
        return 1;
      }

      if (a.x < b.x) {
        return -1;
      }

      return 0;
    });
  }

  private getData(): [][] {
    let result: [][] = [];
    let currentFR = 0;

    // A side effect of this merging is that we could potentially lose framerate entries before the first bitrate entry, I don't think we care
    for (let i = 0; i < this.props.bitrateData.length; i++) {
      let bitrate = this.props.bitrateData[i];
      let thisItem = [bitrate.x, bitrate.y, null, 0];
      // @ts-ignore
      result.push(thisItem);
      for (; currentFR < this.props.framerateData.length; currentFR++) {
        let framerate = this.props.framerateData[currentFR];

        // @ts-ignore
        if (framerate.x.getTime() == thisItem[0].getTime()) {
          thisItem[2] = framerate.y;
        } else if (this.props.bitrateData[i + 1] && framerate.x.getTime() < this.props.bitrateData[i + 1].x.getTime()) {
          thisItem = [framerate.x, null, framerate.y, 0];
          // @ts-ignore
          result.push(thisItem);
        } else {
          break;
        }
      }
    }

    return result;
  }

  private offsetToPercentage(g, offsetX, offsetY) {
    // This is calculating the pixel offset of the leftmost date.
    let xOffset = g.toDomCoords(g.xAxisRange()[0], null)[0];
    let yar0 = g.yAxisRange(0);

    // This is calculating the pixel of the higest value. (Top pixel)
    let yOffset = g.toDomCoords(null, yar0[1])[1];

    // x y w and h are relative to the corner of the drawing area,
    // so that the upper corner of the drawing area is (0, 0).
    let x = offsetX - xOffset;
    let y = offsetY - yOffset;

    // This is computing the rightmost pixel, effectively defining the
    // width.
    let w = g.toDomCoords(g.xAxisRange()[1], null)[0] - xOffset;

    // This is computing the lowest pixel, effectively defining the height.
    let h = g.toDomCoords(null, yar0[0])[1] - yOffset;

    // Percentage from the left.
    let xPct = w === 0 ? 0 : (x / w);
    // Percentage from the top.
    let yPct = h === 0 ? 0 : (y / h);

    // The (1-) part below changes it from "% distance down from the top"
    // to "% distance up from the bottom".
    return [xPct, (1 - yPct)];
  }

  private zoomX(g, zoomInPercentage, xBias) {
    xBias = xBias || 0.5;
    let bounds = g.xAxisExtremes();

    let xAxis = g.xAxisRange();
    let delta = xAxis[1] - xAxis[0];
    let increment = delta * zoomInPercentage;
    let foo = [increment * xBias, increment * (1 - xBias)];

    let min = xAxis[0] + foo[0];
    let max = xAxis[1] - foo[1];

    if (min < bounds[0]) {
      min = bounds[0];
    }

    if (max > bounds[1]) {
      max = bounds[1];
    }

    // Minimum 5 min zoom
    if (max - min < 5 * 60 * 100) {
      return;
    }

    let dateWindow = [min, max];

    g.updateOptions({
      dateWindow,
    });
  }

  private scroll(event, g) {
    let normal = event.mozInputSource ? event.deltaY * -1 : event.deltaY * -1 / 40;
    // For me the normalized value shows 0.075 for one click. If I took
    // that verbatim, it would be a 7.5%.
    let percentage = normal / 50;

    if (!(event.offsetX && event.offsetY)) {
      event.offsetX = event.layerX - event.target.offsetLeft;
      event.offsetY = event.layerY - event.target.offsetTop;
    }

    let percentages = this.offsetToPercentage(g, event.offsetX, event.offsetY);
    let xPct = percentages[0];

    this.zoomX(g, percentage, xPct);
    event.preventDefault();
  }
}
