import * as React from 'react';
import { Redirect } from 'react-router-dom';
import { Data } from 'common/data';
import { DataClassification, PanelItem } from 'components/panel-item/panel-item';
import { SelectInput } from 'components/select-input';
import { TextInput } from 'components/text-input';
import { Button, ButtonType, CoreText, Display, FlexDirection, FormGroupOrientation, JustifyContent, Layout, TextType } from 'twitch-core-ui';

export interface State {
  navigate: string | undefined;
  customerID: string | undefined;
  region: string | undefined;
  contentID: string | undefined;
}

export class SessionViewerHome extends React.Component<{}, State> {
  public state: State = {
    navigate: undefined,
    customerID: undefined,
    region: undefined,
    contentID: undefined,
  };

  public render() {
    if (this.state.navigate !== undefined) {
      return <Redirect to={this.state.navigate} push={true} />;
    }

    return (
      <Layout>
        <Layout padding={{ bottom: 2 }} display={Display.Flex}>
          <Layout flexGrow={1}>
            <CoreText type={TextType.H2}>Session Viewer</CoreText>
          </Layout>
        </Layout>
        <Layout display={Display.Flex} flexDirection={FlexDirection.Column}>
          <Layout padding={{ bottom: 3 }}>
            <TextInput label="Account ID" setValue={this.setCustomerID} orientation={FormGroupOrientation.Horizontal} />
          </Layout>
          <Layout padding={{ bottom: 3 }}>
            <SelectInput label="Region" setValue={this.setRegion} orientation={FormGroupOrientation.Horizontal} options={Data.HomeRegions} />
          </Layout>
          <Layout padding={{ bottom: 3 }}>
            <TextInput label="Channel ID" setValue={this.setContentID} orientation={FormGroupOrientation.Horizontal} />
          </Layout>
        </Layout>
        <Layout display={Display.Flex} justifyContent={JustifyContent.End} padding={{ bottom: 3 }}>
          <Button disabled={!this.isValid(this.state)} onClick={this.getSessions} type={ButtonType.Primary}>Search</Button>
        </Layout>
        <Layout>
          <Layout padding={{ bottom: 2 }}>
            <CoreText type={TextType.H3}>What can I share with the customer?</CoreText>
          </Layout>
          <PanelItem dataClassification={DataClassification.Internal} name="Internal" value="This data is internal and should not be shared with the customer" hideCopy />
          <PanelItem dataClassification={DataClassification.CustomerOnly} name="Customer Only" value="This data can be shared with the customer" hideCopy />
          <PanelItem dataClassification={DataClassification.External} name="External" value="This data can be ascertained by anyone with access to the stream so can be shared with the customer" hideCopy />
        </Layout>
      </Layout>
    );
  }

  public componentDidMount() {
    document.addEventListener('keydown', this.onKeyDown);
  }

  public componentWillUnmount() {
    document.removeEventListener('keydown', this.onKeyDown);
  }

  private onKeyDown = (ev: KeyboardEvent) => {
    if (ev.keyCode == 13 && this.isValid(this.state)) {
      this.getSessions();
    }
  }

  private isValid(state: State): boolean {
    if (!state) {
      return false;
    }

    if (state.customerID === undefined || state.customerID === '') {
      return false;
    }

    if (state.contentID === undefined || state.contentID === '') {
      return false;
    }

    return true;
  }

  private setCustomerID = (customerID: string | undefined): string | undefined => {
    this.setState({
      customerID,
    });

    if (!customerID) {
      return 'Please supply a customer ID';
    }

    return undefined;
  }

  private setRegion = (region: string | undefined): string | undefined => {
    this.setState({
      region,
    });

    if (!region) {
      return 'Please supply a region';
    }

    return undefined;
  }

  private setContentID = (contentID: string | undefined): string | undefined => {
    this.setState({
      contentID,
    });

    if (!contentID) {
      return 'Please supply a customer ID';
    }

    return undefined;
  }

  private getSessions = () => {
    this.setState({
      navigate: '/sessionviewer/' + this.state.customerID + '/' + this.state.region + '/' + this.state.contentID,
    });
  }
}
