import * as React from 'react';
import { match, Redirect } from 'react-router-dom';
import { AppContext } from 'common/appcontext';
import { DataClassification, PanelItem, PanelItemWarning } from 'components/panel-item/panel-item';
import {APIError, TimeStamp} from 'lib/api';
import { submitQuery } from 'modules/reports/api';
import { LimitBreachList, ListLimitBreachEventsResponse, getLimitBreaches } from 'components/limitbreaches'

import {
  Button,
  CoreInteractive,
  CoreText,
  Display,
  FlexDirection,
  FlexWrap,
  FontSize,
  Icon,
  Layout,
  LoadingSpinner,
  StyledLayout,
  SVGAsset,
  TableBody, TableCell, TableRow,
  TextType
} from 'twitch-core-ui';
import { BackupSession, getSECEvents, getSessionDetail, Session, SessionDetail } from '../../api';
import { SECEvent } from 'common/sec/model'
import { TimeseriesChart } from '../../components/session-graph';

interface URLParams {
  customer_id: string;
  region: string;
  content_id: string;
  session_id: string;
  master_session_id: string;
}

interface Props {
  match: match<URLParams>;
}

interface State {
  session?: SessionDetail;
  navigate?: string;
  processing: boolean;
  secEvents?: SECEvent[];
  limitBreaches?: SECEvent[];
}

interface Event {
  time: TimeStamp;
  description: string;
  linkTo?: string;
}

enum ConstraintSet {
  Zero = 128,
  One = 64,
  Two = 32,
  Three = 16,
  Four = 8,
  Five = 4,
}

export class SessionDetailPage extends React.Component<Props, State> {
  public state: State = {
    navigate: undefined,
    session: undefined,
    secEvents: [],
    processing: true,
  };

  public fetchPromise: Promise<SessionDetail | undefined> | undefined;
  public secFetchPromise: Promise<SECEvent[] | undefined> | undefined;
  public limitBreachFetchPromise: Promise<ListLimitBreachEventsResponse | undefined> | undefined;

  public render() {
    if (this.state.navigate !== undefined) {
      // Since we're often navigating to the same component
      this.setState({
        navigate: undefined,
      });
      return <Redirect to={this.state.navigate} push={true} />;
    }

    if (this.state.processing) {
      return (
        <LoadingSpinner />
      );
    }

    if (this.state.session) {
      let sessionName = '';

      if (this.props.match.params.customer_id == 'twitch') {
        sessionName = this.props.match.params.content_id;
      } else {
        sessionName = this.props.match.params.region + '.' + this.props.match.params.customer_id + '.channel.' + this.props.match.params.content_id;
      }

      let timeRange = this.context.time.defaultFormat(this.state.session.ingest_session.stream_up);

      if (this.state.session.ingest_session.stream_down) {
        timeRange += ' - ' + this.context.time.defaultFormat(this.state.session.ingest_session.stream_down);
      } else {
        timeRange += ' - present';
      }

      let graphContent = (<Layout></Layout>);

      if (this.state.session) {
        graphContent = (
          <TimeseriesChart datasetLabel="asasd" bitrateData={this.getBitrateData()} framerateData={this.getFramerateData()} failoverData={this.getFailoverData()} starvationData={this.getStarvationData()} />
        );
      }

      let isPrimarySession = false;
      if (this.state.session && (!this.state.session.ingest_session || !this.state.session.ingest_session.broadcast_format || this.state.session.ingest_session.broadcast_format == 'live')) {
        isPrimarySession = true;
      }

      let backLink = '/sessionviewer/' + this.props.match.params.customer_id + '/' + this.props.match.params.region + '/' + this.props.match.params.content_id;
      let backText = 'Back to session list';

      if (!isPrimarySession) {
        backLink = '/sessionviewer/' + this.props.match.params.customer_id + '/' + this.props.match.params.region + '/' + this.props.match.params.content_id + '/' + this.props.match.params.master_session_id;
        backText = 'Back to primary session';
      }

      let ingestOrigin = (<></>);

      if (this.state.session.transcode_sessions && this.state.session.transcode_sessions.length > 0) {
        // I hate that I have to do this :(
        let fuckYou = this.state.session.transcode_sessions[0].ingest_origin.replace(/^(ingest-(?:[a-z]+-)?[0-9a-f]+)-[0-9]+(\..+)$/, '$1$2.justin.tv.');

        ingestOrigin = (
          <PanelItem dataClassification={DataClassification.Internal} name="Ingest Host" value={fuckYou} />
        );
      }

      let codecRe = /^avc1\.([0-9a-fA-F]{2,2})([0-9a-fA-F]{2,2})([0-9a-fA-F]{2,2})$/;

      let profile = 'Unknown';
      let level = this.state.session.rtmp_session.avc_level / 10;

      if (codecRe.test(this.state.session.rtmp_session.video_codecs)) {
        let data = codecRe.exec(this.state.session.rtmp_session.video_codecs);
        if (data) {
          let profileHex = data[1];
          let constraintHex = data[2];
          let levelHex = data[3];

          profile = this.getProfile(parseInt(profileHex, 16), parseInt(constraintHex, 16));
          level = parseInt(levelHex, 16) / 10;
        }
      }

      let audioChannelWarning: PanelItemWarning | undefined = undefined;
      if (this.state.session.rtmp_session.audio_channels > 2) {
        audioChannelWarning = {
          description: 'Number of audio channels is greater than 2.\nThis can cause playback issues on some devices.',
        };
      }

      let ipInfo = (<></>);
      if (this.state.session.rtmp_session.ip_info) {
        ipInfo = (
          <Layout padding={{ left: 2 }}>
            <PanelItem name="Country" dataClassification={DataClassification.CustomerOnly} value={this.state.session.rtmp_session.ip_info.country} />
            <PanelItem name="City" dataClassification={DataClassification.CustomerOnly} value={this.state.session.rtmp_session.ip_info.city} />
            <PanelItem name="ASN" dataClassification={DataClassification.CustomerOnly} value={this.state.session.rtmp_session.ip_info.asn} />
            <PanelItem name="ISP" dataClassification={DataClassification.CustomerOnly} value={this.state.session.rtmp_session.ip_info.isp_name} />
          </Layout>
        );
      }

      let rtmpInfo = (
        <StyledLayout elevation={2} margin={3} padding={2} flexGrow={1}>
          <Layout padding={{ bottom: 2 }}>
            <CoreText type={TextType.H3}>RTMP Info <CoreInteractive blurAfterClick onClick={() => { this.showRTMPMetadata(this.state.session ? this.state.session.rtmp_session.rtmp_metadata : ''); }}><Icon asset={SVGAsset.Popout} /></CoreInteractive></CoreText>
          </Layout>
          <Layout>
            <PanelItem name="Resolution" dataClassification={DataClassification.External} value={this.getResolution(this.state.session.rtmp_session.video_resolution_width, this.state.session.rtmp_session.video_resolution_height)} />
            <PanelItem name="Encoder" dataClassification={DataClassification.CustomerOnly} value={this.state.session.rtmp_session.encoder} />
            <PanelItem name="Client IP" dataClassification={DataClassification.CustomerOnly} value={this.state.session.rtmp_session.client_ip} />
            {ipInfo}
            <PanelItem name="Video Codecs" dataClassification={DataClassification.External} value={this.state.session.rtmp_session.video_codecs} />
            <PanelItem name="AVC Profile" dataClassification={DataClassification.External} value={profile} />
            <PanelItem name="AVC Level" dataClassification={DataClassification.External} value={level} />
            <PanelItem name="Video Data Rate" dataClassification={DataClassification.External} value={this.state.session.rtmp_session.video_data_rate} />
            <PanelItem name="Video Frame Rate" dataClassification={DataClassification.External} value={this.state.session.rtmp_session.video_frame_rate} />
            <PanelItem name="IDR Interval (ms)" dataClassification={DataClassification.External} value={this.state.session.rtmp_session.idr_interval} />
            <PanelItem name="Segment Duration (ms)" dataClassification={DataClassification.External} value={this.state.session.rtmp_session.segment_duration} />
            <PanelItem name="Audio Codecs" dataClassification={DataClassification.External} value={this.state.session.rtmp_session.audio_codecs} />
            <PanelItem name="Audio Data Rate" dataClassification={DataClassification.External} value={this.state.session.rtmp_session.audio_data_rate} />
            <PanelItem name="Audio Sample Rate" dataClassification={DataClassification.External} value={this.state.session.rtmp_session.audio_sample_rate} />
            <PanelItem name="Audio Sample Size" dataClassification={DataClassification.External} value={this.state.session.rtmp_session.audio_sample_size} />
            <PanelItem name="Audio Channels" dataClassification={DataClassification.External} value={this.state.session.rtmp_session.audio_channels} panelItemWarning={audioChannelWarning} />
            <PanelItem name="Audio Is Stereo" dataClassification={DataClassification.External} value={(!!this.state.session.rtmp_session.audio_is_stereo).toString()} />
            <PanelItem name="RTMP Flags" dataClassification={DataClassification.CustomerOnly} value={this.state.session.rtmp_session.rtmp_flags} />
          </Layout>
        </StyledLayout>
      );

      let transcodeSessions = (<></>);
      if (isPrimarySession && this.state.session.transcode_sessions) {
        transcodeSessions = (
          <>
            <Layout padding={{ bottom: 2 }}>
              <CoreText type={TextType.H3}>Transcode Sessions</CoreText>
            </Layout>
            <Layout>
              <Layout>
                {this.state.session.transcode_sessions.map((tsess, index) => (
                  <Layout key={index}>
                    <PanelItem dataClassification={DataClassification.Internal} name={this.context.time.defaultFormat(tsess.timestamp)} value={tsess.host + '.justin.tv.'} />
                    <Layout padding={{ left: 2 }}>
                      <PanelItem dataClassification={DataClassification.Internal} name="Profile" value={tsess.profile} />
                      <PanelItem dataClassification={DataClassification.Internal} name="Recording S3 Bucket" value={tsess.recording_s3_bucket} />
                      <PanelItem dataClassification={DataClassification.Internal} name="Recording Prefix" value={tsess.recording_prefix} />
                    </Layout>
                  </Layout>
                ))}
              </Layout>
            </Layout>
          </>
        );
      }

      let eventsOutput = (<></>);
      let events = this.getEvents();

      if (events.length > 0) {
        eventsOutput = (
          <StyledLayout elevation={2} margin={3} padding={2} flexGrow={1}>
            <Layout padding={{ bottom: 2 }}>
              <CoreText type={TextType.H3}>Log Events</CoreText>
            </Layout>
            <Layout padding={{ bottom: 2 }}>
              {events.map((event, index) => (
                <Layout key={index}>
                  <PanelItem dataClassification={DataClassification.CustomerOnly} name={this.context.time.defaultFormat(event.time)} value={event.description} linkTo={event.linkTo} hideCopy />
                </Layout>
              ))}
              <PanelItem dataClassification={DataClassification.CustomerOnly} name="RTMP Exit Reason" value={this.state.session.rtmp_session.rtmp_exit_reason} />
            </Layout>
          </StyledLayout>
        );
      }

      let secEventsOutput = (<></>);

      if (this.state.secEvents && this.state.secEvents.length > 0) {
        secEventsOutput = (
          <StyledLayout elevation={2} margin={3} padding={2} flexGrow={1}>
            <Layout padding={{ bottom: 2 }}>
              <CoreText type={TextType.H3}>SEC Events</CoreText>
            </Layout>
            <Layout padding={{ bottom: 2 }}>
              {this.state.secEvents.map((event, index) => (
                <Layout key={index}>
                  <PanelItem dataClassification={DataClassification.CustomerOnly} name={this.context.time.defaultFormat(event.event_time)} value={event.event_name} hideCopy />
                </Layout>
              ))}
            </Layout>
          </StyledLayout>
        );
      }

      let limitBreachesOutput = (<></>);

      if (this.state.limitBreaches && this.state.limitBreaches.length > 0) {
        limitBreachesOutput = (
            <StyledLayout elevation={2} margin={3} padding={2} flexGrow={1}>
              <Layout padding={{ bottom: 2 }}>
                <CoreText type={TextType.H3}>Limit Breaches</CoreText>
              </Layout>
              <Layout padding={{ bottom: 2 }}>
                {this.state.limitBreaches.map((event, index) => (
                    <Layout key={index}>
                      <PanelItem
                          dataClassification={DataClassification.CustomerOnly}
                          name={this.context.time.defaultFormat(event.event_time)}
                          value={event.event_name + ' - ' + LimitBreachList.formatDetail(event)} hideCopy
                      />
                    </Layout>
                ))}
              </Layout>
            </StyledLayout>
        );
      }

      let relatedSessionsOutput = (<></>);
      if (isPrimarySession && this.state.session && this.state.session.related_sessions) {
        relatedSessionsOutput = (
          <StyledLayout elevation={2} margin={3} padding={2} flexGrow={1}>
            <Layout padding={{ bottom: 2 }}>
              <CoreText type={TextType.H3}>Backup Sessions</CoreText>
            </Layout>
            <Layout padding={{ bottom: 2 }}>
              {this.state.session.related_sessions.map((rSess, index) => (
                <Layout key={index}>
                  <PanelItem
                    dataClassification={DataClassification.CustomerOnly}
                    name={rSess.broadcast_format}
                    value={this.formatRelatedSession(rSess)}
                    linkTo={'/sessionviewer/' + this.props.match.params.customer_id + '/' + this.props.match.params.region + '/' + this.props.match.params.content_id + '/' + this.props.match.params.session_id + '/' + rSess.session_id}
                    hideCopy
                  />
                </Layout>
              ))}
            </Layout>
          </StyledLayout>
        );
      }

      return (
        <Layout display={Display.Flex} flexDirection={FlexDirection.Column}>
          <Layout display={Display.Flex} flexDirection={FlexDirection.Row} padding={{ bottom: 5 }}>
            <Layout flexGrow={1}>
              <CoreText type={TextType.H2}> Broadcast Session for {sessionName}: {timeRange}</CoreText>
            </Layout>
            <Layout display={Display.Flex}>
              <Layout margin={{ left: 1 }}>
                <Button linkTo={backLink}>{backText}</Button>
              </Layout>
              <Layout margin={{ left: 1 }}>
                <Button onClick={this.generateSessionDetailReport}>Generate Session Detail Report</Button>
              </Layout>
            </Layout>
          </Layout>
          <Layout flexGrow={4} padding={{ bottom: 2 }} flexShrink={0}>
            {graphContent}
          </Layout>
          <Layout display={Display.Flex} flexDirection={FlexDirection.Row} flexWrap={FlexWrap.Wrap}>
            {limitBreachesOutput}
            {rtmpInfo}
            <StyledLayout elevation={2} margin={3} padding={2} flexGrow={1}>
              <Layout padding={{ bottom: 2 }}>
                <CoreText type={TextType.H3}>Ingest Info</CoreText>
              </Layout>
              <Layout padding={{ bottom: 2 }}>
                <PanelItem name="Ingest Proxy" dataClassification={DataClassification.Internal} value={this.state.session.ingest_session.ingest_host} />
                <PanelItem name="Ingest Cluster" dataClassification={DataClassification.CustomerOnly} value={this.state.session.ingest_session.ingest_proxy} />
                {ingestOrigin}
                <PanelItem name="Broadcast Format" dataClassification={DataClassification.CustomerOnly} value={this.state.session.ingest_session.broadcast_format} />
              </Layout>
              {transcodeSessions}
            </StyledLayout>
            {secEventsOutput}
            {eventsOutput}
            {relatedSessionsOutput}
          </Layout>
        </Layout>
      );
    }

    return (
      <div>We didn't get the session for some reason</div>
    );
  }

  public componentDidMount() {
    this.getSessionDetail();
    this.getSECEvents();
    this.getLimitBreaches();
  }

  public componentDidUpdate(prevProps: Props) {
    if (this.props.match.params.session_id === prevProps.match.params.session_id) {
      return;
    }

    this.setState({
      processing: true,
    });

    this.componentDidMount()
  }

  private formatRelatedSession(rSess: Session): string {
    let timeRange = this.context.time.defaultFormat(rSess.start);

    if (rSess.end) {
      timeRange += ' - ' + this.context.time.defaultFormat(rSess.end);
    }

    return timeRange;
  }

  private getEvents(): Event[] {
    let data: Event[] = [];

    if (this.state.session && this.state.session.ingest_session) {
      data.push({
        time: this.state.session.ingest_session.stream_up,
        description: 'Stream started',
      });
    }

    if (this.state.session && this.state.session.ingest_session && this.state.session.ingest_session.stream_down) {
      data.push({
        time: this.state.session.ingest_session.stream_down,
        description: 'Stream ended',
      });
    }

    if (this.state.session && this.state.session.backup_ingest_session && this.state.session.backup_ingest_session.stitched_to) {
      for (let backupSession of this.state.session.backup_ingest_session.stitched_to) {
        data.push({
          time: backupSession.stitched_timestamp,
          description: 'Failed over to ' + backupSession.broadcast_format,
          linkTo: '/sessionviewer/' + this.props.match.params.customer_id + '/' + this.props.match.params.region + '/' + this.props.match.params.content_id + '/' + this.props.match.params.session_id + '/' + backupSession.session_id,
        });
      }
    }

    if (this.state.session && this.state.session.starvations) {
      for (let starvation of this.state.session.starvations) {
        data.push({
          time: starvation.start,
          description: 'Starvation started',
        });
        if (starvation.duration && starvation.duration.seconds) {
          data.push({
            time: {seconds: starvation.start.seconds + starvation.duration.seconds} as TimeStamp,
            description: 'Starvation ended',
          });
        }
      }
    }

    return data.sort((a: Event, b: Event) => {
      if (a.time > b.time) {
        return 1;
      }

      if (a.time < b.time) {
        return -1;
      }

      return 0;
    });
  }

  private getResolution(width: number, height: number): string {
    if (!width) {
      width = 0;
    }

    if (!height) {
      height = 0;
    }

    return width.toString() + ' x ' + height.toString();
  }

  private getProfile(profileVal: number, constraintSetVal: number): string {
    let profile = 'Unknown';

    /* tslint:disable no-bitwise */
    switch (profileVal) {
      case 44:
        profile = 'CAVLC 4:4:4 Intra';
        break;
      case 66:
        if ((constraintSetVal & ConstraintSet.One) == ConstraintSet.One) {
          profile = 'Constrained Baseline';
        } else {
          profile = 'Baseline';
        }
        break;
      case 77:
        profile = 'Main';
        break;
      case 83:
        if ((constraintSetVal & ConstraintSet.Five) == ConstraintSet.Five) {
          profile = 'Scalable Constrained Baseline';
        } else {
          profile = 'Scalable Baseline';
        }
        break;
      case 86:
        if ((constraintSetVal & ConstraintSet.Five) == ConstraintSet.Five) {
          profile = 'Scalable Constrained High';
        } else if ((constraintSetVal & ConstraintSet.Three) == ConstraintSet.Three) {
          profile = 'Scalable High Intra';
        } else {
          profile = 'Scalable High';
        }
        break;
      case 88:
        profile = 'Extended';
        break;
      case 100:
        if ((constraintSetVal & ConstraintSet.Four) == ConstraintSet.Four) {
          profile = 'Progressive High';
        } else if ((constraintSetVal & (ConstraintSet.Four | ConstraintSet.Five)) == (ConstraintSet.Four | ConstraintSet.Five)) {
          profile = 'Constrained High';
        } else {
          profile = 'High';
        }
        break;
      case 110:
        if ((constraintSetVal & ConstraintSet.Three) == ConstraintSet.Three) {
          profile = 'High 10 Intra';
        } else {
          profile = 'High 10';
        }
        break;
      case 118:
        profile = 'Multiview High';
        break;
      case 122:
        if ((constraintSetVal & ConstraintSet.Three) == ConstraintSet.Three) {
          profile = 'High 4:2:2 Intra';
        } else {
          profile = 'High 4:2:2';
        }
        break;
      case 128:
        profile = 'Stereo High';
        break;
      case 134:
        profile = 'MFC High';
        break;
      case 135:
        profile = 'MFC Depth High';
        break;
      case 138:
        profile = 'Multiview Depth High';
        break;
      case 139:
        profile = 'Enhanced Multiview Depth High';
        break;
      case 244:
        if ((constraintSetVal & ConstraintSet.Three) == ConstraintSet.Three) {
          profile = 'High 4:4:4 Intra';
        } else {
          profile = 'High 4:4:4 Predictive';
        }

        break;
    }
    /* tslint:enable no-bitwise */

    return profile;
  }

  private getBitrateData(): { x: Date, y: number }[] {
    let data: any[] = [];

    if (this.state.session && this.state.session.bitrates) {
      for (let bitrate of this.state.session.bitrates) {
        data.push({
          x: this.context.time.timeStampToDate(bitrate.timestamp),
          y: bitrate.value_kbps,
        });
      }
    }

    return data;
  }

  private getFailoverData(): { x: Date, y: BackupSession, onClick: () => void }[] {
    let data: any[] = [];

    if (this.state.session && this.state.session.backup_ingest_session && this.state.session.backup_ingest_session.stitched_to) {
      for (let backupSession of this.state.session.backup_ingest_session.stitched_to) {
        data.push({
          x: this.context.time.timeStampToDate(backupSession.stitched_timestamp),
          y: backupSession,
          onClick: () => {
            this.setState({
              navigate: '/sessionviewer/' + this.props.match.params.customer_id + '/' + this.props.match.params.region + '/' + this.props.match.params.content_id + '/' + this.props.match.params.session_id + '/' + backupSession.session_id,
            });
          },
        });
      }
    }

    return data;
  }

  private getStarvationData(): { x: Date, y: string }[] {
    let data: any[] = [];

    if (this.state.session && this.state.session.starvations) {
      for (let starvation of this.state.session.starvations) {
        data.push({
          x: this.context.time.timeStampToDate(starvation.start),
          y: 'start',
        });
        data.push({
          x: this.context.time.timeStampToDate({seconds: starvation.start.seconds + starvation.duration.seconds} as TimeStamp),
          y: 'end',
        });
      }
    }

    return data;
  }

  private getFramerateData(): { x: Date, y: number }[] {
    let data: any[] = [];

    if (this.state.session && this.state.session.framerates) {
      for (let framerate of this.state.session.framerates) {
        data.push({
          x: this.context.time.timeStampToDate(framerate.timestamp),
          y: framerate.value_fps,
        });
      }
    }

    return data;
  }

  private showRTMPMetadata(metadata: string) {
    this.context.showModal(
      (
        <CoreText fontSize={FontSize.Size4}>{metadata}</CoreText>
      ),
      {
        title: 'Raw RTMP Metadata',
        onClose: () => {
          this.context.hideModal();
        },
      },
    );
  }

  private generateSessionDetailReport = () => {
    if (!this.state.session) {
      return;
    }

    // These dates don't have to be modified for time zones because they are converted to UTC before sending to the server
    let sessionEnd = new Date();

    if (this.state.session.ingest_session.stream_down) {
      sessionEnd = new Date(this.state.session.ingest_session.stream_down.seconds * 1000);
    }

    submitQuery({
      query_type: 'broadcast_session_info',
      start: new Date(this.state.session.ingest_session.stream_up.seconds * 1000),
      end: sessionEnd,
      broadcast_session_id: this.state.session.session_id,
    }).then((queryID: string) => {
      this.setState({
        navigate: '/reports/' + queryID,
      });
    }, (reason: APIError) => {
      this.context.showModal(
        (
          <div>{reason.message}</div>
        ),
        {
          title: 'Error Generating Report',
          onClose: () => {
            this.context.hideModal();
          },
        },
      );
    });
  }

  private async getSECEvents() {
    if (this.secFetchPromise) {
      Promise.reject(this.secFetchPromise);
      this.secFetchPromise = undefined;
    }

    this.secFetchPromise = getSECEvents(this.props.match.params.customer_id, this.props.match.params.region, this.props.match.params.content_id, this.props.match.params.session_id);

    await this.secFetchPromise.then((events) => {
      this.secFetchPromise = undefined;
      this.setState({
        secEvents: events,
      });
    });
  }

  private async getLimitBreaches() {
    if (this.limitBreachFetchPromise) {
      Promise.reject(this.limitBreachFetchPromise);
      this.limitBreachFetchPromise = undefined;
    }

    this.limitBreachFetchPromise = getLimitBreaches(this.props.match.params.customer_id, this.props.match.params.region, this.props.match.params.session_id, '');

    await this.limitBreachFetchPromise.then((response) => {
      this.limitBreachFetchPromise = undefined;
      this.setState({
        limitBreaches: response.limit_breach_events,
      });
    });
  }

  private async getSessionDetail() {
    if (this.fetchPromise) {
      Promise.reject(this.fetchPromise);
      this.fetchPromise = undefined;
    }

    this.fetchPromise = getSessionDetail(this.props.match.params.customer_id, this.props.match.params.region, this.props.match.params.content_id, this.props.match.params.session_id);

    await this.fetchPromise.then((session) => {
      this.fetchPromise = undefined;
      this.setState({
        session,
        processing: false,
      });
    }, (reason: APIError) => {
      this.context.showModal(
        (
          <CoreText fontSize={FontSize.Size4}>{reason.message}</CoreText>
        ),
        {
          title: 'Error Getting Session List',
          onClose: () => {
            this.context.hideModal();
          },
        },
      );
    });
  }
}

SessionDetailPage.contextType = AppContext;
