import * as React from 'react';
import { Link, match } from 'react-router-dom';
import { AppContext } from 'common/appcontext';
import { APIError, TimeStamp } from 'lib/api';
import { CoreText, Layout, LoadingSpinner, Table, TableBody, TableCell, TableHeader, TableHeading, TableRow, TextType } from 'twitch-core-ui';
import { getSessions, Session } from '../../api';
import { Paging, Position } from 'components/paging';

interface URLParams {
  customer_id: string;
  region: string;
  content_id: string;
}

interface Props {
  match: match<URLParams>;
}

interface State {
  sessions?: Session[];
  processing: boolean;
  next_page_token?: string;
  page_stack: string[];
}

export class SessionList extends React.Component<Props, State> {
  public state: State = {
    processing: true,
    page_stack: [],
  };

  public render() {
    if (this.state.processing) {
      return (
        <LoadingSpinner />
      );
    }

    let sessionName = '';

    if (this.props.match.params.customer_id == 'twitch') {
      sessionName = this.props.match.params.content_id;
    } else {
      sessionName = this.props.match.params.region + '.' + this.props.match.params.customer_id + '.channel.' + this.props.match.params.content_id;
    }

    let list = <></>;

    if (this.state && this.state.sessions) {
      list = (
        <TableBody>
          {this.state.sessions.map((session, index) => (
            <TableRow key={index}>
              <TableCell>{this.context.time.defaultFormat(session.start)}</TableCell>
              <TableCell>{this.formatSessionEnd(session.end)}</TableCell>
              <TableCell>{this.formatSessionDuration(session.start, session.end)}</TableCell>
              <TableCell>{session.ingest_edge}</TableCell>
              <TableCell><Link to={'/sessionviewer/' + this.props.match.params.customer_id + '/' + this.props.match.params.region + '/' + this.props.match.params.content_id + '/' + session.session_id}>Details</Link>{this.getRelatedSessions(session)}</TableCell>
            </TableRow>
          ))}
        </TableBody>
      );
    }

    return (
      <Layout>
        <Layout padding={{ bottom: 3 }}>
          <CoreText type={TextType.H2}>Session List for {sessionName}</CoreText>
        </Layout>
        <Layout>
        <Paging loading={this.state.processing} position={Position.Before} prevPageExists={this.state.page_stack.length > 0} nextPageExists={!!this.state.next_page_token} triggerNextPage={() => this.nextPage()} triggerPrevPage={() => this.prevPage()}>
          <Table>
            <TableHeader>
              <TableHeading label="Session Start" />
              <TableHeading label="Session End" />
              <TableHeading label="Duration" />
              <TableHeading label="Ingest Server" />
              <TableHeading label="Details" />
            </TableHeader>
            {list}
          </Table>
        </Paging>
        </Layout>
      </Layout>
    );
  }

  private nextPage() {
    if (!this.state) {
      return
    }

    let ps = this.state.page_stack;

    if (ps.length == 0 || ps[ps.length - 1] != this.state.next_page_token) {
      ps.push(this.state.next_page_token)
    }

    this.setState({
      page_stack: ps,
    })

    this.getSessions(this.state.next_page_token);
  }

  private prevPage() {
    let ps = this.state.page_stack;
    ps.pop()

    this.setState({
      page_stack: ps,
    })

    let page_token = '';

    if (ps.length > 0){
      page_token = ps[ps.length - 1]
    }

    this.getSessions(page_token);
  }

  private formatSessionDuration(start: TimeStamp, end: TimeStamp): string {
    if (!end) {
      return this.context.time.duration(start, {seconds: new Date().getTime() / 1000} as TimeStamp)
    }

    return this.context.time.duration(start, end);
  }

  private formatSessionEnd(end: TimeStamp): string {
    if (!end) {
      return 'LIVE'
    }

    return this.context.time.defaultFormat(end)
  }

  public componentWillMount() {
    this.getSessions('');
  }

  public getRelatedSessions(session: Session) {
    if (session.related_sessions && session.related_sessions.length > 0) {
      return ' (+' + session.related_sessions.length + ' related sessions)';
    }

    return '';
  }

  private async getSessions(pageToken: string) {
    this.setState({
      processing: true,
    });

    await getSessions(this.props.match.params.customer_id, this.props.match.params.region, this.props.match.params.content_id, pageToken).then((sessions) => {
      let nextPageToken = '';
      if (sessions && sessions.length > 0) {
        nextPageToken = sessions[sessions.length - 1].start.seconds.toString()
      }

      this.setState({
        sessions,
        next_page_token: nextPageToken,
        processing: false,
      });
    }, (reason: APIError) => {
      this.setState({
        processing: false,
      });

      this.context.showModal(
        (
          <div>{reason.message}</div>
        ),
        {
          title: 'Error Getting Session List',
          onClose: () => {
            this.context.hideModal();
          },
        },
      );
    });
  }
}

SessionList.contextType = AppContext;
