import {API} from 'lib/api';
import {TimeStamp} from 'lib/api';

export interface AvailableCaptureEndpointRegions {
    regions: string[];
}

export interface CreateCaptureEndpointRequest {
    region: string;
    name: string;
    max_duration_seconds: number;
}

export interface GetCaptureEndpointRequest {
    id: string;
}

export enum CaptureEndpointStatus {
    Initialised = 0,
    Ready = 1,
    Failed = 2,
    Recording = 3,
    Stopped = 4,
}

export interface CaptureEndpointInfo {
    region: string;
    id: string;
    name: string;
    status: number;
    expires: TimeStamp;
}

export interface CaptureEndpointDetail {
    info: CaptureEndpointInfo;
    rtmp_url: string;
    files: CapturedFileInfo[];
}

export interface CapturedFileInfo {
    key: string;
    last_modified: TimeStamp;
    size: number;
}

export interface ListCaptureEndpointsResponse {
    endpoints: CaptureEndpointInfo[];
}

export interface CapturedFileDetail {
    size: number;
    duration: number;
    overall_bitrate: number;
    framerate: number;
    num_video_tracks: number;
    num_audio_tracks: number;
    video_detail: CFDVideoDetail;
    audio_detail: CFDAudioDetail;
    can_download: boolean;
}

export interface CFDVideoDetail {
    format: string;
    profile: string;
    level: number;
    cabac: boolean;
    ref_frames: number;
    codec_id: number;
    bitrate_nominal: number;
    width: number;
    width_sampled: number;
    height: number;
    height_sampled: number;
    pixel_aspect_ratio: number;
    display_aspect_ratio: number;
    framerate_mode: string;
    framerate_original: number;
    framerate: number;
    frame_count: number;
    colour_space: string;
    chroma_subsampling: string;
    bit_depth: number;
    scan_type: string;
    encoder_library: string;
    encoder_settings: Record<string, string>;
}

export interface CFDAudioDetail {
    format: string;
    additional_features: string;
    codec_id: string;
    channels: number;
    channel_positions: string;
    channel_layout: string;
    samples_per_frame: number;
    sample_rate: number;
    sample_count: number;
    framerate: number;
    compression_mode: string;
    delay: number;
    delay_source: string;
}

export interface GOPInfo {
    frames: GOPInfoFrame[];
}

export interface GOPInfoFrame {
    timestamp: number;
    pict_type: string;
    size: number;
}

export interface GetCapturedFileDetailResponse {
    detail: CapturedFileDetail;
    gop: GOPInfo;
}

export async function getAvailableCaptureEndpointRegions(): Promise<AvailableCaptureEndpointRegions> {
    const path = `/api/videoanalysis/capture-endpoint-regions`;

    const response = await API.get<AvailableCaptureEndpointRegions>(path, {});

    if (response.error) {
        return Promise.reject(response.error);
    }

    if (response.body === undefined) {
        return Promise.reject('Empty Response');
    }

    return response.body;
}

export async function createCaptureEndpoint(region: string, name: string, maxCaptureDuration: string): Promise<CaptureEndpointInfo> {
    const path = `/api/videoanalysis/capture-endpoint`;

    let request: CreateCaptureEndpointRequest = {
        region,
        name,
        max_duration_seconds: parseInt(maxCaptureDuration),
    };

    const response = await API.post<CaptureEndpointInfo>(path, {
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify(request),
    });

    if (response.error) {
        return Promise.reject(response.error);
    }

    if (response.body === undefined) {
        return Promise.reject('Empty Response');
    }

    return response.body;
}

export async function getCaptureEndpoint(endpointID: string): Promise<CaptureEndpointDetail> {
    const path = `/api/videoanalysis/capture-endpoint/` + endpointID;

    const response = await API.get<CaptureEndpointDetail>(path, {});

    if (response.error) {
        return Promise.reject(response.error);
    }

    if (response.body === undefined) {
        return Promise.reject('Empty Response');
    }

    return response.body;
}

export async function listCaptureEndpoints(): Promise<ListCaptureEndpointsResponse> {
    const path = `/api/videoanalysis/capture-endpoint`;

    const response = await API.get<ListCaptureEndpointsResponse>(path, {});

    if (response.error) {
        return Promise.reject(response.error);
    }

    if (response.body === undefined) {
        return Promise.reject('Empty Response');
    }

    return response.body;
}

export async function getCapturedFileDetail(key: string): Promise<GetCapturedFileDetailResponse> {
    const path = `/api/videoanalysis/captured-file/` + key;

    const response = await API.get<GetCapturedFileDetailResponse>(path, {});

    if (response.error) {
        return Promise.reject(response.error);
    }

    if (response.body === undefined) {
        return Promise.reject('Empty Response');
    }

    return response.body;
}

export interface GetCapturedFileDownloadLinkResponse {
    download_link: string;
}

export async function getCapturedFileDownloadLink(key: string): Promise<GetCapturedFileDownloadLinkResponse> {
    const path = `/api/videoanalysis/captured-file/link/` + key;

    const response = await API.get<GetCapturedFileDownloadLinkResponse>(path, {});

    if (response.error) {
        return Promise.reject(response.error);
    }

    if (response.body === undefined) {
        return Promise.reject('Empty Response');
    }

    return response.body;
}

export interface CreateRTMPDumpResponse {
    rtmp_dump_id: string;
}

export interface CreateRTMPDumpRequest {
    region: string;
    customer_id: string;
    channel_id: string;
    name: string;
    format: string;
}

export async function createRTMPDump(name: string, region: string, customerID: string, channelID: string, format: string): Promise<CreateRTMPDumpResponse> {
    const path = `/api/videoanalysis/rtmp-dump`;

    let request: CreateRTMPDumpRequest = {
        name,
        region,
        customer_id: customerID,
        channel_id: channelID,
        format,
    };

    const response = await API.post<CreateRTMPDumpResponse>(path, {
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify(request),
    });

    if (response.error) {
        return Promise.reject(response.error);
    }

    if (response.body === undefined) {
        return Promise.reject('Empty Response');
    }

    return response.body;
}

export interface ListRTMPDumpsResponse {
    rtmp_dumps: RTMPDumpSummary[];
}

export enum RTMPDumpStatus {
    Requested = 0,
    InProgress = 1,
    Complete = 2,
    Failed = 3,
}

export interface RTMPDumpSummary {
    rtmp_dump_id: string;
    name: string;
    file_path: string;
    created_at: TimeStamp;
    status: RTMPDumpStatus;
}

export async function getRTMPDump(rtmpDumpID: string): Promise<RTMPDumpSummary> {
    const path = `/api/videoanalysis/rtmp-dump/` + rtmpDumpID;

    const response = await API.get<RTMPDumpSummary>(path, {});

    if (response.error) {
        return Promise.reject(response.error);
    }

    if (response.body === undefined) {
        return Promise.reject('Empty Response');
    }

    return response.body;
}

export async function listRTMPDumps(): Promise<ListRTMPDumpsResponse> {
    const path = `/api/videoanalysis/rtmp-dump`;

    const response = await API.get<ListRTMPDumpsResponse>(path, {});

    if (response.error) {
        return Promise.reject(response.error);
    }

    if (response.body === undefined) {
        return Promise.reject('Empty Response');
    }

    return response.body;
}
