import * as React from 'react';
import { AppContext } from 'common/appcontext';
import { formatBits, formatBytes } from 'common/utils/bytes';
import { DataClassification, PanelItem } from 'components/panel-item/panel-item';
import { APIError } from 'lib/api';
// @ts-ignore
import { AlignItems, AlignSelf, Button, ButtonType, CoreText, Display, FlexDirection, FormGroupOrientation, JustifyContent, Layout, LoadingSpinner, StyledLayout, SVGAsset, Table, TableBody, TableCell, TableHeader, TableHeading, TableRow, TextType } from 'twitch-core-ui';
import { getCapturedFileDetail, GetCapturedFileDetailResponse, getCapturedFileDownloadLink } from '../../api';
import { GOPGraph } from '../../components/gop-graph';

interface Props {
  file_path: string;
}

export interface State {
  loading: boolean;
  gettingDownloadLink: boolean;
  // For capture endpoint list
  file_detail: GetCapturedFileDetailResponse | undefined;
}

export class CapturedFileAnalysis extends React.Component<Props, State> {
  public state: State = {
    loading: false,
    gettingDownloadLink: false,
    file_detail: undefined,
  };

  private reloadTimer: NodeJS.Timer | undefined;

  public render() {
    if (this.state.loading) {
      return <LoadingSpinner />;
    }

    if (!this.state.file_detail || !this.state.file_detail.detail) {
      this.reloadTimer = setTimeout(() => this.getCapturedFileInfo(), 5000);

      return (
        <LoadingSpinner />
      );
    }

    let basicInfo = (
      <StyledLayout elevation={2} margin={3} padding={2} flexGrow={1}>
        <Layout padding={{ bottom: 2 }}>
          <CoreText type={TextType.H3}>Basic Info</CoreText>
        </Layout>
        <Layout>
          <PanelItem name="Size" dataClassification={DataClassification.CustomerOnly} value={formatBytes(this.state.file_detail.detail.size || 0)} />
          <PanelItem name="Duration" dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.duration} />
          <PanelItem name="Overall Bitrate" dataClassification={DataClassification.CustomerOnly} value={formatBits(this.state.file_detail.detail.overall_bitrate || 0) + '/s'} />
          <PanelItem name="Frame Rate" dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.framerate} />
          <PanelItem name="Video Tracks" dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.num_video_tracks} />
          <PanelItem name="Audio Tracks" dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.num_audio_tracks} />
        </Layout>
      </StyledLayout>
    );

    let videoInfo = (
      <StyledLayout elevation={2} margin={3} padding={2} flexGrow={1}>
        <Layout padding={{ bottom: 2 }}>
          <CoreText type={TextType.H3}>Video Info</CoreText>
        </Layout>
        <Layout>
          <PanelItem name="Format" dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.video_detail.format} />
          <PanelItem name="Profile" dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.video_detail.profile} />
          <PanelItem name="Level" dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.video_detail.level} />
          <PanelItem name="CABAC" dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.video_detail.cabac ? 'Yes' : 'No'} />
          <PanelItem name="Ref Frames" dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.video_detail.ref_frames} />
          <PanelItem name="Codec ID" dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.video_detail.codec_id} />
          <PanelItem name="Bitrate Nominal" dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.video_detail.bitrate_nominal ? formatBits(this.state.file_detail.detail.video_detail.bitrate_nominal || 0) + '/s' : this.state.file_detail.detail.video_detail.bitrate_nominal} />
          <PanelItem name="Width" dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.video_detail.width} />
          <PanelItem name="Sampled Width" dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.video_detail.width_sampled} />
          <PanelItem name="Height" dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.video_detail.height} />
          <PanelItem name="Sampled Height" dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.video_detail.height_sampled} />
          <PanelItem name="Pixel Aspect Ratio" dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.video_detail.pixel_aspect_ratio} />
          <PanelItem name="Display Aspect Ratio" dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.video_detail.display_aspect_ratio} />
          <PanelItem name="Framerate Mode" dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.video_detail.framerate_mode} />
          <PanelItem name="Framerate Original" dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.video_detail.framerate_original} />
          <PanelItem name="Framerate" dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.video_detail.framerate} />
          <PanelItem name="Frame Count" dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.video_detail.frame_count} />
          <PanelItem name="Colour Space" dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.video_detail.colour_space} />
          <PanelItem name="Chroma Subsampling" dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.video_detail.chroma_subsampling} />
          <PanelItem name="Bit Depth" dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.video_detail.bit_depth} />
          <PanelItem name="Scan Type" dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.video_detail.scan_type} />
        </Layout>
      </StyledLayout>
    );

    let audioInfo = (
      <StyledLayout elevation={2} margin={3} padding={2} flexGrow={1}>
        <Layout padding={{ bottom: 2 }}>
          <CoreText type={TextType.H3}>Audio Info</CoreText>
        </Layout>
        <Layout>
          <PanelItem name="Format" dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.audio_detail.format} />
          <PanelItem name="Additional Features" dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.audio_detail.additional_features} />
          <PanelItem name="Codec ID" dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.audio_detail.codec_id} />
          <PanelItem name="Channels" dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.audio_detail.channels} />
          <PanelItem name="Channel Positions" dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.audio_detail.channel_positions} />
          <PanelItem name="Channel Layout" dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.audio_detail.channel_layout} />
          <PanelItem name="Smaples per Frame" dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.audio_detail.samples_per_frame} />
          <PanelItem name="Sample Rate" dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.audio_detail.sample_rate} />
          <PanelItem name="Sample Count" dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.audio_detail.sample_count} />
          <PanelItem name="Framerate" dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.audio_detail.framerate} />
          <PanelItem name="Compression Mode" dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.audio_detail.compression_mode} />
          <PanelItem name="Delay" dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.audio_detail.delay} />
          <PanelItem name="Delay Source" dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.audio_detail.delay_source} />
        </Layout>
      </StyledLayout>
    );

    let encoderSettings = (<></>);

    if (this.state.file_detail.detail.video_detail.encoder_settings && Object.keys(this.state.file_detail.detail.video_detail.encoder_settings).length > 0) {
      let encSettings: JSX.Element[] = [
        (<PanelItem name="Encoder Library" dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.video_detail.encoder_library} />),
      ];

      for (let encSetting in this.state.file_detail.detail.video_detail.encoder_settings) {
        encSettings.push(
          <PanelItem name={encSetting} dataClassification={DataClassification.CustomerOnly} value={this.state.file_detail.detail.video_detail.encoder_settings[encSetting]} />,
        );
      }

      encoderSettings = (
        <StyledLayout elevation={2} margin={3} padding={2} flexGrow={1}>
          <Layout padding={{ bottom: 2 }}>
            <CoreText type={TextType.H3}>Encoder Settings</CoreText>
          </Layout>
          <Layout>
            {encSettings}
          </Layout>
        </StyledLayout>
      );
    }

    let gopInfo = (<LoadingSpinner />);

    if (this.state.file_detail.gop) {
      gopInfo = (
        <GOPGraph gopData={this.state.file_detail.gop.frames} />
      );
    } else {
      this.reloadTimer = setTimeout(() => this.getCapturedFileInfo(), 5000);
    }

    let downloadButton = (<></>);

    if (this.state.file_detail.detail.can_download) {
      if (this.state.gettingDownloadLink) {
        downloadButton = (
          <LoadingSpinner />
        );
      } else {
        downloadButton = (
          <Layout>
            <Button onClick={() => this.downloadFile()}>Download File</Button>
          </Layout>
        );
      }
    }

    return (
      <Layout>
        <Layout padding={{ bottom: 2 }} display={Display.Flex} justifyContent={JustifyContent.End}>
          <Layout flexGrow={1}>
            <CoreText type={TextType.H3}>File analysis</CoreText>
          </Layout>
          {downloadButton}
        </Layout>
        {gopInfo}
        <Layout display={Display.Flex}>
          {basicInfo}
          {videoInfo}
          {audioInfo}
          {encoderSettings}
        </Layout>
      </Layout>
    );
  }

  public componentDidUpdate(prevProps: Props) {
    if (this.props.file_path === prevProps.file_path) {
      return;
    }

    this.getCapturedFileInfo();
  }

  public componentDidMount() {
    this.getCapturedFileInfo();
  }

  private downloadFile = () => {
    this.setState({
      gettingDownloadLink: true,
    });
    getCapturedFileDownloadLink(this.props.file_path).then((result) => {
      this.setState({
        gettingDownloadLink: false,
      });
      window.open(result.download_link);
    }, (reason: APIError) => {
      this.setState({
        gettingDownloadLink: false,
      });
      this.context.showModal(
        (
          <div>{reason.message}</div>
        ),
        {
          title: 'Error Getting Download Link',
          onClose: () => {
            this.context.hideModal();
          },
        },
      );
    });
  }

  private getCapturedFileInfo = () => {
    if (this.reloadTimer) {
      clearTimeout(this.reloadTimer);
      this.reloadTimer = undefined;
    }

    this.setState({ loading: true, file_detail: undefined });
    getCapturedFileDetail(this.props.file_path).then((result) => {
      this.setState({
        file_detail: result,
        loading: false,
      });
    });
  }
}

CapturedFileAnalysis.contextType = AppContext;
