import 'dygraphs/dist/dygraph.css';
import Dygraph from 'dygraphs/index.es5';
import * as React from 'react';
import { Layout } from 'twitch-core-ui';
import { GOPInfoFrame } from '../../api';

interface Props {
  gopData: GOPInfoFrame[];
}

export class GOPGraph extends React.Component<Props> {
  public chart?: Dygraph;
  private container: HTMLElement | null;

  public render() {
    return (
      <Layout>
        <div style={{ width: '100%' }} ref={(el) => (this.container = el)} />
      </Layout>
    );
  }

  public componentDidUpdate(prevProps: Props) {
    if (!this.chart || (this.props.gopData === prevProps.gopData)) {
      return;
    }

    let data = this.getData();

    if (this.chart) {
      this.chart.updateOptions({
        file: data,
      });
    }
  }

  public componentDidMount() {
    let data = this.getData();

    if (!this.container) {
      return;
    }

    this.chart = new Dygraph(this.container, data, {
      labels: ['timestamp', 'I', 'P', 'B'],
      legend: 'follow',
      labelsShowZeroValues: false,
      showRangeSelector: true,
      includeZero: true,
      digitsAfterDecimal: 0,
      rollPeriod: 1,
      plotter: this.multiColumnBarPlotter,
      ylabel: 'Frame Size (bytes)',
      xlabel: 'Timestamp (ms)',
      //@ts-ignore
      legendFormatter: (asdasd) => { return this.legendFormatter(asdasd); },
      axes: {
        y: {
          labelsKMG2: true,
        },
      },
      series: {
        I: {
          axis: 'y1',
          showInRangeSelector: true,
          strokeWidth: 2,
        },
        P: {
          axis: 'y1',
          showInRangeSelector: true,
          strokeWidth: 2,
        },
        B: {
          axis: 'y1',
          showInRangeSelector: true,
          strokeWidth: 2,
        },
      },
      interactionModel: {
        wheel: (e, g) => {
          return this.scroll(e, g);
        },
      },
    });
  }

  private legendFormatter(data) {
    if (data.x == null) {
      // This happens when there's no selection and {legend: 'always'} is set.
      return '<br>' + data.series.map((series) => { return series.dashHTML + ' ' + series.labelHTML; }).join('<br>');
    }

    let html = 'Timestamp: ' + data.xHTML;
    data.series.forEach((series) => {
      if (!series.isVisible) { return; }
      html += '<br>' + 'Frame Type: ' + series.dashHTML + ' ' + series.labelHTML;
      html += '<br>' + 'Frame Size: ' + series.yHTML + ' (' + series.y + ')';
    });
    return html;
  }

  // Multiple column bar chart
  private multiColumnBarPlotter(e) {
    // We need to handle all the series simultaneously.
    if (e.seriesIndex !== 0) { return; }

    let g = e.dygraph;
    let ctx = e.drawingContext;
    let sets = e.allSeriesPoints;
    let yBottom = e.dygraph.toDomYCoord(0);

    // Find the minimum separation between x-values.
    // This determines the bar width.
    let minSep = Infinity;
    for (let points of sets) {
      for (let i = 1; i < points.length; i++) {
        let sep = points[i].canvasx - points[i - 1].canvasx;
        if (sep < minSep) { minSep = sep; }
      }
    }
    let barWidth = Math.floor(2.0 / 3 * minSep);

    let fillColors = [];
    let strokeColors = g.getColors();
    for (let strokeColor of strokeColors) {
      //@ts-ignore
      fillColors.push(strokeColor);
    }

    for (let j = 0; j < sets.length; j++) {
      ctx.fillStyle = fillColors[j];
      ctx.strokeStyle = strokeColors[j];
      for (let p of sets[j]) {
        let centerX = p.canvasx;
        let xLeft = centerX - (barWidth / 2);

        ctx.fillRect(xLeft, p.canvasy, barWidth, yBottom - p.canvasy);
        ctx.strokeRect(xLeft, p.canvasy, barWidth, yBottom - p.canvasy);
      }
    }
  }

  private getData(): [][] {
    let result: [][] = [];

    if (!this.props.gopData) {
      return result;
    }

    for (let pict of this.props.gopData) {
      let thisItem = [pict.timestamp || 0, null, null, null];
      switch (pict.pict_type) {
        case 'I':
          thisItem[1] = pict.size;
          break;
        case 'P':
          thisItem[2] = pict.size;
          break;
        case 'B':
          thisItem[3] = pict.size;
          break;
      }
      // @ts-ignore
      result.push(thisItem);
    }

    return result;
  }

  private offsetToPercentage(g, offsetX, offsetY) {
    // This is calculating the pixel offset of the leftmost date.
    let xOffset = g.toDomCoords(g.xAxisRange()[0], null)[0];
    let yar0 = g.yAxisRange(0);

    // This is calculating the pixel of the higest value. (Top pixel)
    let yOffset = g.toDomCoords(null, yar0[1])[1];

    // x y w and h are relative to the corner of the drawing area,
    // so that the upper corner of the drawing area is (0, 0).
    let x = offsetX - xOffset;
    let y = offsetY - yOffset;

    // This is computing the rightmost pixel, effectively defining the
    // width.
    let w = g.toDomCoords(g.xAxisRange()[1], null)[0] - xOffset;

    // This is computing the lowest pixel, effectively defining the height.
    let h = g.toDomCoords(null, yar0[0])[1] - yOffset;

    // Percentage from the left.
    let xPct = w === 0 ? 0 : (x / w);
    // Percentage from the top.
    let yPct = h === 0 ? 0 : (y / h);

    // The (1-) part below changes it from "% distance down from the top"
    // to "% distance up from the bottom".
    return [xPct, (1 - yPct)];
  }

  private zoomX(g, zoomInPercentage, xBias) {
    xBias = xBias || 0.5;
    let bounds = g.xAxisExtremes();

    let xAxis = g.xAxisRange();
    let delta = xAxis[1] - xAxis[0];
    let increment = delta * zoomInPercentage;
    let foo = [increment * xBias, increment * (1 - xBias)];

    let min = xAxis[0] + foo[0];
    let max = xAxis[1] - foo[1];

    if (min < bounds[0]) {
      min = bounds[0];
    }

    if (max > bounds[1]) {
      max = bounds[1];
    }

    // Minimum 5 min zoom
    if (max - min < 5 * 60 * 100) {
      return;
    }

    let dateWindow = [min, max];

    g.updateOptions({
      dateWindow,
    });
  }

  private scroll(event, g) {
    let normal = event.mozInputSource ? event.deltaY * -1 : event.deltaY * -1 / 40;
    // For me the normalized value shows 0.075 for one click. If I took
    // that verbatim, it would be a 7.5%.
    let percentage = normal / 50;

    if (!(event.offsetX && event.offsetY)) {
      event.offsetX = event.layerX - event.target.offsetLeft;
      event.offsetY = event.layerY - event.target.offsetTop;
    }

    let percentages = this.offsetToPercentage(g, event.offsetX, event.offsetY);
    let xPct = percentages[0];

    this.zoomX(g, percentage, xPct);
    event.preventDefault();
  }
}
