import * as React from 'react';
import {Link, match, Redirect} from 'react-router-dom';
import {AppContext} from 'common/appcontext';
import {formatBytes} from 'common/utils/bytes';
import {DataClassification, PanelItem} from 'components/panel-item/panel-item';
// @ts-ignore
import {
  AlignSelf,
  Button,
  ButtonType,
  CoreText,
  Display,
  FlexDirection,
  FormGroupOrientation,
  JustifyContent,
  Layout,
  LoadingSpinner,
  StyledLayout,
  SVGAsset,
  Table,
  TableBody,
  TableCell,
  TableHeader,
  TableHeading,
  TableRow,
  TextType
} from 'twitch-core-ui';
import {CaptureEndpointDetail, CaptureEndpointStatus, getCaptureEndpoint} from '../../api';

interface URLParams {
  endpoint_id: string;
}

interface Props {
  match: match<URLParams>;
}

export interface State {
  navigate: string | undefined;
  loading: boolean;
  // For capture endpoint list
  capture_endpoint: CaptureEndpointDetail | undefined;
}

export class CaptureEndpoint extends React.Component<Props, State> {
  public state: State = {
    navigate: undefined,
    loading: false,
    capture_endpoint: undefined,
  };

  public reloadTimer: NodeJS.Timer | undefined;

  public render() {
    if (this.state.navigate !== undefined) {
      return <Redirect to={this.state.navigate} push={true}/>;
    }

    if (this.state.loading) {
      return <LoadingSpinner/>;
    }

    let endpointInfo = <></>;

    if (this.state && this.state.capture_endpoint && this.state.capture_endpoint.info) {
      endpointInfo = (
          <Layout margin={{bottom: 3}}>
            <Layout padding={{bottom: 2}}>
              <CoreText type={TextType.H3}>Endpoint Info</CoreText>
            </Layout>
            <StyledLayout elevation={2} padding={2} flexGrow={1}>
              <Layout padding={{bottom: 2}}>
                <PanelItem name="Name" dataClassification={DataClassification.Internal} value={this.state.capture_endpoint.info.name}/>
                <PanelItem name="Region" dataClassification={DataClassification.Internal} value={this.state.capture_endpoint.info.region}/>
                <PanelItem name="RTMP URL" dataClassification={DataClassification.CustomerOnly} value={this.state.capture_endpoint.rtmp_url}/>
                <PanelItem name="Status" dataClassification={DataClassification.CustomerOnly}
                           value={CaptureEndpointStatus[this.state.capture_endpoint.info.status ? this.state.capture_endpoint.info.status : 0]}/>
                <PanelItem name="Expires" dataClassification={DataClassification.CustomerOnly}
                           value={this.context.time.defaultFormat(this.state.capture_endpoint.info.expires)}/>
              </Layout>
            </StyledLayout>
          </Layout>
      );
    }

    let files = <></>;

    if (this.state && this.state.capture_endpoint && this.state.capture_endpoint.files && this.state.capture_endpoint.files.length > 0) {
      files = (
          <Layout>
            <Layout padding={{bottom: 2}}>
              <CoreText type={TextType.H3}>Captured Files</CoreText>
            </Layout>
            <Table>
              <TableHeader>
                <TableHeading label="ID"/>
                <TableHeading label="Created"/>
                <TableHeading label="Size"/>
                <TableHeading label="Details"/>
              </TableHeader>
              <TableBody>
                {this.state.capture_endpoint.files.map((file, index) => (
                    <TableRow key={index}>
                      <TableCell>{file.key}</TableCell>
                      <TableCell>{this.context.time.since(file.last_modified)}</TableCell>
                      <TableCell>{formatBytes(file.size || 0)}</TableCell>
                      <TableCell><Link to={'/videoanalysis/captured-file/' + file.key}>Details</Link></TableCell>
                    </TableRow>
                ))}
              </TableBody>
            </Table>
          </Layout>
      );
    }

    return (
        <Layout>
          {endpointInfo}
          {files}
        </Layout>
    );
  }

  public componentDidMount() {
    this.getCaptureEndpoint(true);
  }

  private getCaptureEndpoint = (showSpinner: boolean) => {
    if (this.reloadTimer) {
      clearTimeout(this.reloadTimer);
      this.reloadTimer = undefined;
    }

    this.setState({
      loading: showSpinner,
    });

    getCaptureEndpoint(this.props.match.params.endpoint_id).then((result) => {
      if (result.info && result.info.status == CaptureEndpointStatus.Ready && !result.rtmp_url) {
        result.rtmp_url = 'Awaiting...';
      }

      if (result.info && (
          result.info.status == CaptureEndpointStatus.Initialised ||
          result.info.status == CaptureEndpointStatus.Ready ||
          result.info.status == CaptureEndpointStatus.Recording)) {
        this.reloadTimer = setTimeout(() => this.getCaptureEndpoint(false), 5000);
      }

      this.setState({
        capture_endpoint: result,
        loading: false,
      });
    });
  }
}

CaptureEndpoint.contextType = AppContext;
