import * as React from 'react';
import {Link, Redirect} from 'react-router-dom';
import {AppContext} from 'common/appcontext';
// @ts-ignore
import {
    AlignSelf,
    Button,
    ButtonType,
    CoreText,
    Display,
    FlexDirection,
    FontSize,
    FormGroupOrientation,
    JustifyContent,
    Layout,
    SVGAsset,
    Table,
    TableBody,
    TableCell,
    TableHeader,
    TableHeading,
    TableRow,
    TextType
} from 'twitch-core-ui';
import {CaptureEndpointInfo, CaptureEndpointStatus, listCaptureEndpoints, listRTMPDumps, RTMPDumpStatus, RTMPDumpSummary} from '../../api';
import {TimeStamp} from "lib/api";

export interface State {
    navigate: string | undefined;
    loading: boolean;
    // For capture endpoint list
    capture_endpoints: CaptureEndpointInfo[];
    rtmp_dumps: RTMPDumpSummary[];
}

export class VideoAnalysisHome extends React.Component<{}, State> {
    public state: State = {
        navigate: undefined,
        loading: false,
        capture_endpoints: [],
        rtmp_dumps: [],
    };

    public render() {
        if (this.state.navigate !== undefined) {
            return <Redirect to={this.state.navigate} push={true}/>;
        }

        let endpointList = <></>;

        if (this.state && this.state.capture_endpoints) {
            endpointList = (
                <TableBody>
                    {this.state.capture_endpoints.map((endpoint, index) => (
                        <TableRow key={index}>
                            <TableCell>{endpoint.name}</TableCell>
                            <TableCell>{endpoint.id}</TableCell>
                            <TableCell>{CaptureEndpointStatus[endpoint.status ? endpoint.status : 0]}</TableCell>
                            <TableCell>{this.formatExpiry(endpoint.expires)}</TableCell>
                            <TableCell><Link to={'/videoanalysis/capture-endpoint/' + endpoint.id}>Details</Link></TableCell>
                        </TableRow>
                    ))}
                </TableBody>
            );
        }

        let dumpList = <></>;

        if (this.state && this.state.rtmp_dumps) {
            dumpList = (
                <TableBody>
                    {this.state.rtmp_dumps.map((dump, index) => (
                        <TableRow key={index}>
                            <TableCell>{dump.name}</TableCell>
                            <TableCell>{this.context.time.since(dump.created_at)}</TableCell>
                            <TableCell>{dump.rtmp_dump_id}</TableCell>
                            <TableCell>{RTMPDumpStatus[dump.status ? dump.status : 0]}</TableCell>
                            <TableCell><Link to={'/videoanalysis/rtmp-dump/' + dump.rtmp_dump_id}>Details</Link></TableCell>
                        </TableRow>
                    ))}
                </TableBody>
            );
        }

        return (
            <Layout>
                <Layout margin={{bottom: 4}}>
                    <Layout margin={{bottom: 2}}>
                        <CoreText type={TextType.H2}>Video Analysis</CoreText>
                    </Layout>
                    <Layout>
                        <CoreText type={TextType.P} fontSize={FontSize.Size4}>
                            This module allows you to analyse in progress streams as well as providing specific test endpoints that can be given to
                            customers to capture incoming RTMP data for the purposes of analysis and testing.
                        </CoreText>
                        <CoreText type={TextType.P} fontSize={FontSize.Size4}>
                            Capture endpoints automatically expire after 48 hours and the files that are produced, as well as RTMP dumps, are kept for
                            7 days
                        </CoreText>
                    </Layout>
                </Layout>
                <Layout margin={{bottom: 4}}>
                    <Layout padding={{bottom: 2}} display={Display.Flex}>
                        <Layout flexGrow={1}>
                            <CoreText type={TextType.H3}>RTMP Dumps</CoreText>
                        </Layout>
                        <Layout alignSelf={AlignSelf.End}>
                            <Button icon={SVGAsset.Plus} linkTo="/videoanalysis/rtmp-dump/new">New</Button>
                        </Layout>
                    </Layout>
                    <Layout>
                        <Table>
                            <TableHeader>
                                <TableHeading label="Name"/>
                                <TableHeading label="Created At"/>
                                <TableHeading label="ID"/>
                                <TableHeading label="Status"/>
                                <TableHeading label="Details"/>
                            </TableHeader>
                            {dumpList}
                        </Table>
                    </Layout>
                </Layout>
                <Layout margin={{bottom: 4}}>
                    <Layout padding={{bottom: 2}} display={Display.Flex}>
                        <Layout flexGrow={1}>
                            <CoreText type={TextType.H3}>RTMP Capture Endpoints</CoreText>
                        </Layout>
                        <Layout alignSelf={AlignSelf.End}>
                            <Button icon={SVGAsset.Plus} linkTo="/videoanalysis/capture-endpoint/new">New</Button>
                        </Layout>
                    </Layout>
                    <Layout>
                        <Table>
                            <TableHeader>
                                <TableHeading label="Name"/>
                                <TableHeading label="ID"/>
                                <TableHeading label="Status"/>
                                <TableHeading label="Expires"/>
                                <TableHeading label="Details"/>
                            </TableHeader>
                            {endpointList}
                        </Table>
                    </Layout>
                </Layout>
            </Layout>
        );
    }

    public componentDidMount() {
        this.listCaptureEndpoints();
    }

    private formatExpiry = (expires: TimeStamp): string => {
        let now = new Date().getTime() / 1000

        if (expires.seconds > now) {
            return this.context.time.duration({seconds: now}, expires, '', 'in ', true)
        }

        return this.context.time.since(expires)
    }

    private listCaptureEndpoints = () => {
        listCaptureEndpoints().then((result) => {
            this.setState({capture_endpoints: result.endpoints});
        });

        listRTMPDumps().then((result) => {
            this.setState({rtmp_dumps: result.rtmp_dumps});
        });
    }
}

VideoAnalysisHome.contextType = AppContext;
