import * as React from 'react';
import { Redirect } from 'react-router-dom';
import { AppContext } from 'common/appcontext';
import { Option, SelectInput } from 'components/select-input';
import { TextInput } from 'components/text-input';
import { APIError } from 'lib/api';
// @ts-ignore
import { AlignSelf, Button, ButtonType, CoreText, Display, FlexDirection, FormGroupOrientation, JustifyContent, Layout, LoadingSpinner, SVGAsset, Table, TableBody, TableCell, TableHeader, TableHeading, TableRow, TextType } from 'twitch-core-ui';
import { CaptureEndpointInfo, createCaptureEndpoint, getAvailableCaptureEndpointRegions } from '../../api';

export interface State {
  navigate: string | undefined;
  loading: boolean;
  // For capture endpoint list
  region?: string;
  maxCaptureDuration?: string;
  name?: string;
  capture_endpoint_regions: Option[];
}

export class NewCaptureEndpoint extends React.Component<{}, State> {
  public state: State = {
    navigate: undefined,
    loading: false,
    capture_endpoint_regions: [],
  };

  public static CaptureDurations: Option[] = [
    {
      label: '10m',
      value: '600',
      selected: true,
    },
    {
      label: '1 hr',
      value: '3600',
      selected: false,
    },
    {
      label: '6 hrs',
      value: '21600',
      selected: false,
    },
  ];

  public render() {
    if (this.state.navigate !== undefined) {
      return <Redirect to={this.state.navigate} push={true} />;
    }

    if (this.state.loading) {
      return <LoadingSpinner />;
    }

    return (
      <Layout>
        <Layout padding={{ bottom: 2 }} display={Display.Flex}>
          <Layout flexGrow={1}>
            <CoreText type={TextType.H2}>New Capture Endpoint</CoreText>
          </Layout>
        </Layout>
        <Layout padding={{ bottom: 3 }}>
          <SelectInput label="Region" setValue={this.setRegion} defaultValue={undefined} orientation={FormGroupOrientation.Horizontal} options={this.state.capture_endpoint_regions} />
        </Layout>
        <Layout padding={{ bottom: 3 }}>
          <TextInput label="Name" setValue={this.setName} orientation={FormGroupOrientation.Horizontal} />
        </Layout>
        <Layout padding={{ bottom: 3 }}>
          <SelectInput label="Max Capture Duration" setValue={this.setMaxCaptureDuration} orientation={FormGroupOrientation.Horizontal} options={NewCaptureEndpoint.CaptureDurations} />
        </Layout>
        <Layout display={Display.Flex} justifyContent={JustifyContent.End} padding={{ bottom: 3 }}>
          <Button disabled={!this.isValid(this.state)} onClick={this.createEndpoint} type={ButtonType.Primary}>Submit</Button>
        </Layout>
      </Layout>
    );
  }

  public componentDidMount() {
    document.addEventListener('keydown', this.onKeyDown);

    this.getAvailableCaptureEndpointRegions();
  }

  public componentWillUnmount() {
    document.removeEventListener('keydown', this.onKeyDown);
  }

  private setRegion = (region: string | undefined): string | undefined => {
    this.setState({
      region,
    });

    if (!region) {
      return 'Please supply a region';
    }

    return undefined;
  }

  private setMaxCaptureDuration = (maxCaptureDuration: string | undefined): string | undefined => {
    this.setState({
      maxCaptureDuration,
    });

    if (!maxCaptureDuration) {
      return 'Please supply a max capture duration';
    }

    return undefined;
  }



  private setName = (name: string | undefined): string | undefined => {
    this.setState({
      name,
    });

    if (!name) {
      return 'Please supply a name';
    }

    return undefined;
  }

  private onKeyDown = (ev: KeyboardEvent) => {
    if (ev.keyCode == 13 && this.isValid(this.state)) {
      this.createEndpoint();
    }
  }

  private isValid(state: State): boolean {
    if (!state) {
      return false;
    }

    if (!state.region || state.region == 'new_endpoint') {
      return false;
    }

    if (!state.name) {
      return false;
    }

    return true;
  }

  private createEndpoint = () => {
    if (!this.isValid(this.state)) {
      return;
    }

    this.setState({
      loading: true,
    });

    //@ts-ignore
    createCaptureEndpoint(this.state.region, this.state.name, this.state.maxCaptureDuration).then((endpoint: CaptureEndpointInfo) => {
      this.setState({
        navigate: '/videoanalysis/capture-endpoint/' + endpoint.id,
        loading: false,
      });
    }, (reason: APIError) => {
      this.setState({
        loading: false,
      });

      this.context.showModal(
        (
          <div>{reason.message}</div>
        ),
        {
          title: 'Error Creating Capture Endpoint',
          onClose: () => {
            this.context.hideModal();
          },
        },
      );
    });
  }

  private getAvailableCaptureEndpointRegions = () => {
    getAvailableCaptureEndpointRegions().then((result) => {
      let regionList: Option[] = [
        {
          label: 'Select Region',
          value: 'new_endpoint',
          selected: true,
        },
      ];

      for (let region of result.regions) {
        regionList.push({
          label: region,
          value: region,
          selected: false,
        });
      }

      this.setState({
        capture_endpoint_regions: regionList,
      });
    });
  }
}

NewCaptureEndpoint.contextType = AppContext;
