import * as React from 'react';
import {Redirect} from 'react-router-dom';
import {AppContext} from 'common/appcontext';
import {Data} from 'common/data';
import {SelectInput} from 'components/select-input';
import {TextInput} from 'components/text-input';
import {APIError} from 'lib/api';
// @ts-ignore
import {
    AlignSelf,
    Button,
    ButtonType,
    CoreText,
    Display,
    FlexDirection,
    FormGroupOrientation,
    JustifyContent,
    Layout,
    LoadingSpinner,
    SVGAsset,
    Table,
    TableBody,
    TableCell,
    TableHeader,
    TableHeading,
    TableRow,
    TextType
} from 'twitch-core-ui';
import {createRTMPDump, CreateRTMPDumpResponse} from '../../api';

export interface State {
    navigate: string | undefined;
    loading: boolean;
    name: string | undefined;
    region: string | undefined;
    customerID: string | undefined;
    channelID: string | undefined;
    format: string | undefined;
}

export class NewRTMPDump extends React.Component<{}, State> {
    public state: State = {
        navigate: undefined,
        loading: false,
        name: undefined,
        region: 'us-west-2',
        customerID: undefined,
        channelID: undefined,
        format: 'live',
    };

    public render() {
        if (this.state.navigate !== undefined) {
            return <Redirect to={this.state.navigate} push={true}/>;
        }

        if (this.state.loading) {
            return <LoadingSpinner/>;
        }

        return (
            <Layout>
                <Layout padding={{bottom: 2}} display={Display.Flex}>
                    <Layout flexGrow={1}>
                        <CoreText type={TextType.H2}>New RTMP Dump</CoreText>
                    </Layout>
                </Layout>
                <Layout padding={{bottom: 3}}>
                    <TextInput label="Name" setValue={this.setName} defaultValue={this.state.name} orientation={FormGroupOrientation.Horizontal}/>
                </Layout>
                <Layout padding={{bottom: 3}}>
                    <SelectInput label="Region" setValue={this.setRegion} defaultValue={this.state.region}
                                 orientation={FormGroupOrientation.Horizontal} options={Data.HomeRegions}/>
                </Layout>
                <Layout padding={{bottom: 3}}>
                    <TextInput label="Customer ID" setValue={this.setCustomerID} defaultValue={this.state.customerID}
                               orientation={FormGroupOrientation.Horizontal}/>
                </Layout>
                <Layout padding={{bottom: 3}}>
                    <TextInput label="Channel ID" setValue={this.setChannelID} defaultValue={this.state.channelID}
                               orientation={FormGroupOrientation.Horizontal}/>
                </Layout>
                <Layout padding={{bottom: 3}}>
                    <TextInput label="Format" setValue={this.setFormat} defaultValue={this.state.format}
                               orientation={FormGroupOrientation.Horizontal}/>
                </Layout>
                <Layout display={Display.Flex} justifyContent={JustifyContent.End} padding={{bottom: 3}}>
                    <Button disabled={!this.isValid(this.state)} onClick={this.createRTMPDump} type={ButtonType.Primary}>Submit</Button>
                </Layout>
            </Layout>
        );
    }

    public componentDidMount() {
        document.addEventListener('keydown', this.onKeyDown);
    }

    public componentWillUnmount() {
        document.removeEventListener('keydown', this.onKeyDown);
    }

    private setRegion = (region: string | undefined): string | undefined => {
        this.setState({
            region,
        });

        if (!region) {
            return 'Please supply a region';
        }

        return undefined;
    }

    private setName = (name: string | undefined): string | undefined => {
        this.setState({
            name,
        });

        if (!name) {
            return 'Please supply a name';
        }

        return undefined;
    }

    private setCustomerID = (customerID: string | undefined): string | undefined => {
        this.setState({
            customerID,
        });

        if (!customerID) {
            return 'Please supply a customer ID';
        }

        return undefined;
    }

    private setChannelID = (channelID: string | undefined): string | undefined => {
        this.setState({
            channelID,
        });

        if (!channelID) {
            return 'Please supply a channel ID';
        }

        return undefined;
    }

    private setFormat = (format: string | undefined): string | undefined => {
        this.setState({
            format,
        });

        if (!format) {
            return 'Please supply a format';
        }

        return undefined;
    }

    private onKeyDown = (ev: KeyboardEvent) => {
        if (ev.keyCode == 13 && this.isValid(this.state)) {
            this.createRTMPDump();
        }
    }

    private isValid(state: State): boolean {
        if (!state) {
            return false;
        }

        if (!state.region) {
            return false;
        }

        if (!state.name) {
            return false;
        }

        if (!state.customerID) {
            return false;
        }

        if (!state.channelID) {
            return false;
        }

        if (!state.format) {
            return false;
        }

        return true;
    }

    private createRTMPDump = () => {
        if (!this.isValid(this.state)) {
            return;
        }

        this.setState({
            loading: true,
        });

        // @ts-ignore
        createRTMPDump(this.state.name, this.state.region, this.state.customerID, this.state.channelID, this.state.format).then((result: CreateRTMPDumpResponse) => {
            this.setState({
                navigate: '/videoanalysis/rtmp-dump/' + result.rtmp_dump_id,
                loading: false,
            });
        }, (reason: APIError) => {
            this.setState({
                loading: false,
            });
            this.context.showModal(
                (
                    <div>{reason.message}</div>
                ),
                {
                    title: 'Error Creating RTMP Dump',
                    onClose: () => {
                        this.context.hideModal();
                    },
                },
            );
        });
    }
}

NewRTMPDump.contextType = AppContext;
