import * as React from 'react';
import { match, Redirect } from 'react-router-dom';
import { AppContext } from 'common/appcontext';
import { DataClassification, PanelItem } from 'components/panel-item/panel-item';
// @ts-ignore
import { AlignSelf, Button, ButtonType, CoreText, Display, FlexDirection, FormGroupOrientation, JustifyContent, Layout, LoadingSpinner, StyledLayout, SVGAsset, Table, TableBody, TableCell, TableHeader, TableHeading, TableRow, TextType } from 'twitch-core-ui';
import { getRTMPDump, RTMPDumpStatus, RTMPDumpSummary } from '../../api';
import { CapturedFileAnalysis } from '../../components/captured-file';

interface URLParams {
  rtmp_dump_id: string;
}

interface Props {
  match: match<URLParams>;
}

export interface State {
  navigate: string | undefined;
  loading: boolean;
  // For capture endpoint list
  rtmp_dump: RTMPDumpSummary | undefined;
}

export class RTMPDump extends React.Component<Props, State> {
  public state: State = {
    navigate: undefined,
    loading: false,
    rtmp_dump: undefined,
  };

  public render() {
    if (this.state.navigate !== undefined) {
      return <Redirect to={this.state.navigate} push={true} />;
    }

    if (this.state.loading) {
      return <LoadingSpinner />;
    }


    let dumpInfo = <></>;

    if (this.state && this.state.rtmp_dump) {
      dumpInfo = (
        <Layout margin={{ bottom: 3 }}>
          <Layout padding={{ bottom: 2 }}>
            <CoreText type={TextType.H3}>RTMP Dump Info</CoreText>
          </Layout>
          <StyledLayout elevation={2} padding={2} flexGrow={1}>
            <Layout padding={{ bottom: 2 }}>
              <PanelItem name="Name" dataClassification={DataClassification.Internal} value={this.state.rtmp_dump.name} />
              <PanelItem name="File Path" dataClassification={DataClassification.Internal} value={this.state.rtmp_dump.file_path} />
              <PanelItem name="Created At" dataClassification={DataClassification.CustomerOnly} value={this.context.time.since(this.state.rtmp_dump.created_at)} />
              <PanelItem name="Status" dataClassification={DataClassification.CustomerOnly} value={RTMPDumpStatus[this.state.rtmp_dump.status ? this.state.rtmp_dump.status : 0]} />
            </Layout>
          </StyledLayout>
        </Layout>
      );
    }

    let fileAnalysis = (<></>);

    if (this.state.rtmp_dump && this.state.rtmp_dump.status == RTMPDumpStatus.Complete) {
      fileAnalysis = (
        <CapturedFileAnalysis file_path={this.state.rtmp_dump.file_path} />
      );
    }

    return (
      <Layout>
        {dumpInfo}
        {fileAnalysis}
      </Layout>
    );
  }

  public componentDidMount() {
    this.getRTMPDump(true);
  }

  private getRTMPDump = (showSpinner: boolean) => {
    this.setState({
      loading: showSpinner,
    });

    getRTMPDump(this.props.match.params.rtmp_dump_id).then((result) => {
      this.setState({
        rtmp_dump: result,
        loading: false,
      });

      let dumpStatus = result.status || 0;

      if (dumpStatus == RTMPDumpStatus.Requested || dumpStatus == RTMPDumpStatus.InProgress) {
        setTimeout(() => this.getRTMPDump(false), 5000);
      }
    });
  }
}

RTMPDump.contextType = AppContext;
