package main

import (
	"context"
	"github.com/Sirupsen/logrus"
	"github.com/aws/aws-lambda-go/events"
	"github.com/aws/aws-lambda-go/lambda"
	"github.com/aws/aws-sdk-go/service/ssm"

	"code.justin.tv/coleiain/api-gateway-handler"
	"code.justin.tv/event-engineering/streamtester/pkg/aws"
	awsBackend "code.justin.tv/event-engineering/streamtester/pkg/aws/backend"
	"code.justin.tv/event-engineering/streamtester/pkg/streamtester-api"
)

var (
	logger             = logrus.New()
	handler            *streamtesterAPI.Handler
	awsClient          *aws.Client
	binaryContentTypes = []string{"application/protobuf"}
	withDecrpytion     = true
)

func init() {
	logger.SetLevel(logrus.DebugLevel)

	awsBackend, err := awsBackend.New()
	if err != nil {
		panic(err)
	}

	awsClient = aws.New(awsBackend, logger)
	params, err := getParamsData(logger)
	if err != nil {
		panic(err)
	}
	certPem := []byte(params["/streamtester/certPem"])
	keyPem := []byte(params["/streamtester/keyPem"])
	caPem := []byte(params["/streamtester/caPem"])

	handler, err = streamtesterAPI.New(certPem, keyPem, caPem)

	if err != nil {
		panic(err)
	}
}

func main() {
	lambda.Start(Handle)
}

// Handle responds to lambda triggers and executes code accordingly
func Handle(ctx context.Context, request events.APIGatewayProxyRequest) (events.APIGatewayProxyResponse, error) {
	return apiGatewayHandler.ProcessRequest(ctx, request, binaryContentTypes, handler.Server)
}

func getParamsData(logger *logrus.Logger) (map[string]string, error) {
	params := map[string]string{
		"/streamtester/certPem": "",
		"/streamtester/keyPem":  "",
		"/streamtester/caPem":   "",
	}

	// Grab the values from SSM
	paramNames := make([]*string, 0)
	for k, _ := range params {
		theParam := k
		paramNames = append(paramNames, &theParam)
	}

	ssmResponse, err := awsClient.SSMGetParameters(&ssm.GetParametersInput{
		Names:          paramNames,
		WithDecryption: &withDecrpytion,
	})

	if err != nil {
		return nil, err
	}

	for _, param := range ssmResponse.Parameters {
		for k, _ := range params {
			if k == *param.Name {
				params[k] = *param.Value
			}
		}
	}

	return params, nil
}
