package main

import (
	streamtester "code.justin.tv/event-engineering/streamtester/pkg/streamtester"
	"github.com/Sirupsen/logrus"
	"github.com/spf13/cobra"
	"os"
	"regexp"
)

// APIURL is set at build time
var APIURL string

// PrecompiledAccessKey can be set at build time to set a default access key in the binary
var PrecompiledAccessKey string

// PrecompiledRTMPURL can be set at build time to set a default RTMP URL in the binary
var PrecompiledRTMPURL string

var contentIDRegex = regexp.MustCompile("[^A-Za-z0-9\\-\\_]")
var logger *logrus.Logger

var rootCmd = &cobra.Command{
	Use:   "streamtester",
	Short: "Streamtester CLI, run test streams!",
	Long:  ``,
	Run: func(cmd *cobra.Command, args []string) {
		accessKey := cmd.Flag("access-key").Value.String()
		if accessKey == "" {
			logger.Error("Please supply an access key")
			return
		}

		usePCID, err := cmd.Flags().GetBool("pcid")
		if err != nil {
			usePCID = false
		}

		contentID := cmd.Flag("content-id").Value.String()
		if usePCID {
			contentID, err = os.Hostname()
			if err != nil {
				logger.Warn("Error getting host name ", err)
			}
		}

		contentID = contentIDRegex.ReplaceAllString(contentID, "-")

		tester := streamtester.New(APIURL, cmd.Flag("access-key").Value.String(), logger)
		err = tester.Start(contentID, cmd.Flag("source-file").Value.String(), cmd.Flag("rtmp-url").Value.String())
		if err != nil {
			logger.Error("Error starting stream ", err)
		}
	},
}

func init() {
	rootCmd.PersistentFlags().String("access-key", PrecompiledAccessKey, "Access Key")
	rootCmd.PersistentFlags().String("content-id", "", "Content ID")
	rootCmd.PersistentFlags().Bool("pcid", false, "Use PC Name as Content ID")
	rootCmd.PersistentFlags().String("rtmp-url", PrecompiledRTMPURL, "RTMP URL")
	rootCmd.PersistentFlags().String("source-file", "gaming_1080p_6000k.flv", "Source File")

	logger = logrus.New()
	logger.SetLevel(logrus.InfoLevel)
}

func main() {
	if err := rootCmd.Execute(); err != nil {
		logger.Error("Error executing command ", err)
		os.Exit(1)
	}
}
