package backend

import (
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/credentials/ec2rolecreds"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/ssm"
	"github.com/pkg/errors"
)

// Client contains methods for AWS interactions to allow easier data mocking for tests.
type Client interface {
	SSMGetParameters(input *ssm.GetParametersInput) (*ssm.GetParametersOutput, error)
}

type client struct {
	ssm *ssm.SSM
}

// New calls the NewClient function from the base jira package
func New() (Client, error) {
	// Set up S3 Client
	creds := credentials.NewChainCredentials(
		[]credentials.Provider{
			&credentials.EnvProvider{},
			&ec2rolecreds.EC2RoleProvider{},
		})

	sess, err := session.NewSession(&aws.Config{
		Region:      aws.String("us-west-2"),
		Credentials: creds,
	})

	if err != nil {
		return nil, errors.Wrap(err, "Error creating AWS session")
	}

	return &client{
		ssm: ssm.New(sess),
	}, nil
}

// SSMGetParameters calls the underlying GetParameters from the ssm backend
func (c *client) SSMGetParameters(input *ssm.GetParametersInput) (*ssm.GetParametersOutput, error) {
	return c.ssm.GetParameters(input)
}
