package runner

import (
	logging "code.justin.tv/event-engineering/golibs/pkg/logging"
	"code.justin.tv/event-engineering/gortmp/pkg/rtmp"
	"code.justin.tv/event-engineering/rtmp/pkg/flvlooper"
	"github.com/pkg/errors"
	"io"
	"time"
)

func Run(logger logging.Logger, flvFile *flvlooper.File, outputStream rtmp.MediaStream) {
	defer flvFile.Close()

	var thisLoopStartTs uint32
	var fileStartTs uint32
	var lastTs uint32
	ref := time.Now()

	for {
		header, bytes, err := flvFile.ReadTag()

		if err != nil {
			// If err == io.EOF then we're at the end of the file, so loop
			if err == io.EOF {
				flvFile.LoopBack()
				thisLoopStartTs = lastTs
				logger.Debugf("Looping source, resetting timestamp loopStart:%v", thisLoopStartTs)
				ref = time.Now()
				continue
			} else {
				// We got an unexpected error, so we're going to break here
				logger.Warn(errors.Wrap(err, "Unexpected error from flvFile.ReadTag"))
				outputStream.Close()
				break
			}
		}

		tag := &rtmp.FlvTag{
			Type:      header.TagType,
			Timestamp: header.Timestamp,
			Size:      header.DataSize,
			Bytes:     bytes,
		}

		if tag.Timestamp == 0 {
			tag.Timestamp = lastTs
		} else {
			if fileStartTs == 0 {
				fileStartTs = tag.Timestamp
			}

			// Make timestamps relative to 0
			tag.Timestamp = tag.Timestamp - fileStartTs

			// Work out how long we want to pause for, otherwise we just shove things in buffers as fast as we can and things get clogged up
			clock := time.Since(ref)
			target := time.Millisecond * time.Duration(tag.Timestamp)
			waitTime := target - clock

			if waitTime > 500*time.Millisecond {
				// clock drift of >500ms means we need to adjust our reference
				logger.Debugf("adjusting for clock drift of %s", waitTime)
				ref = ref.Add(-waitTime)
			} else if waitTime > 0 {
				time.Sleep(waitTime)
			}

			// Fix timestamps as we're looping
			tag.Timestamp = thisLoopStartTs + tag.Timestamp

			// I'm not sure why this happens sometimes, but I had a source FLV where the last tag in the file had a timestamp that was lower than the previous one
			// Just coding around it for now until I can find out more about if this is valid or I should rework the source media
			if tag.Timestamp < lastTs {
				tag.Timestamp = lastTs
			}

			lastTs = tag.Timestamp

			// Check if it's closed first
			if outputStream.IsClosed() {
				logger.Info("Outputstream closed, ending loop")
				break
			}
		}

		// Push tag to output stream
		err = outputStream.Publish(tag)

		if err != nil {
			logger.Warn(err)
		}
	}
}
