package streamtester

import (
	logging "code.justin.tv/event-engineering/golibs/pkg/logging"
	"code.justin.tv/event-engineering/gortmp/pkg/rtmp"
	flvlooper "code.justin.tv/event-engineering/rtmp/pkg/flvlooper"
	rtmppusher "code.justin.tv/event-engineering/rtmp/pkg/pusher"
	rpc "code.justin.tv/event-engineering/streamtester/pkg/rpc"
	runner "code.justin.tv/event-engineering/streamtester/pkg/runner"
	"context"
	"fmt"
	"github.com/google/uuid"
	"github.com/pkg/errors"
	twirp "github.com/twitchtv/twirp"
	"io"
	"net/http"
	"os"
	"time"
)

var downloadURLPrefix = "http://twitch-event-engineering-public.s3.amazonaws.com/streamtester/%v"
var retryTime = 5 * time.Second

type StreamTester interface {
	Start(contentId, fileName, rtmpEndpoint string) error
}

type backend struct {
	client rpc.StreamTester
	apiKey string
	logger logging.Logger
}

func (b *backend) getTwirpContext() context.Context {
	ctx := context.Background()
	header := make(http.Header)
	header.Set("x-api-key", b.apiKey)

	twContext, err := twirp.WithHTTPRequestHeaders(ctx, header)
	if err != nil {
		b.logger.Warnf("Error creating twirp context %v", err)
	}

	return twContext
}

func New(apiURL, accessKey string, logger logging.Logger) StreamTester {
	httpClient := &http.Client{}

	client := rpc.NewStreamTesterJSONClient(apiURL, httpClient)

	server := &backend{
		client: client,
		apiKey: accessKey,
		logger: logger,
	}

	return server
}

func (b *backend) Start(contentId, fileName, rtmpEndpoint string) error {
	if contentId == "" {
		contentId = uuid.New().String()
	}

	b.logger.Infof("Starting stream with content id %v", contentId)
	b.logger.Infof("Inspect at https://inspector.twitch.tv/#/lvs.twitch-stream-tester.%v", contentId)

	// Grab the stream key from the server
	resp, err := b.client.CreateStreamKey(b.getTwirpContext(), &rpc.CreateStreamKeyReq{
		ContentId: contentId,
	})

	if err != nil {
		b.logger.Warnf("Error getting stream key %v", err)
		return err
	}

	// Grab the file from S3 if it doesn't exist
	var looper *flvlooper.File

	if _, err := os.Stat(fileName); err == nil {
		fileR, fileErr := os.OpenFile(fileName, os.O_RDONLY, 0600)
		if fileErr != nil {
			return errors.Wrap(fileErr, "Error opening cache file for read")
		}

		var lpErr error
		looper, lpErr = flvlooper.Create(fileR)
		if lpErr != nil {
			return errors.Wrap(lpErr, "Error creating FLV source")
		}
	} else if !os.IsNotExist(err) {
		return errors.Wrap(err, "Error opening cached file")
	} else {
		resp, dlErr := http.Get(fmt.Sprintf(downloadURLPrefix, fileName))
		if dlErr != nil {
			return dlErr
		}

		if resp.StatusCode != 200 {
			return fmt.Errorf("Failed to download source file %v", resp.StatusCode)
		}

		fileW, writeErr := os.OpenFile(fileName, os.O_WRONLY|os.O_CREATE, 0600)

		if writeErr != nil {
			return errors.Wrap(writeErr, "Error creating local file")
		}

		reader := io.TeeReader(resp.Body, fileW)

		// We need to read at least some data before kicking off the rest of the reads in a goroutine
		dlbuf := make([]byte, 1024)
		// We don't actually need to do anything with the data read here, TeeReader will handle the writing of this data to the appropriate place
		_, err = reader.Read(dlbuf)
		if err != nil {
			if err == io.EOF {
				fmt.Println("Finished downloading file")
			} else {
				return err
			}
		}

		fileR, err := os.OpenFile(fileName, os.O_RDONLY, 0600)
		if err != nil {
			return errors.Wrap(err, "Error opening cache file for read")
		}

		var lpErr error
		looper, lpErr = flvlooper.Create(fileR)
		if lpErr != nil {
			return errors.Wrap(lpErr, "Error creating FLV source")
		}

		// Download the file in a goroutine
		go func() {
			defer fileW.Close()
			defer resp.Body.Close()
			for {
				_, readErr := reader.Read(dlbuf)
				if readErr != nil {
					if readErr == io.EOF {
						fmt.Println("Finished downloading file")
					} else {
						fmt.Println("Error reading from source stream", readErr)
					}
					return
				}
			}
		}()
	}

	b.runOutput(looper, fmt.Sprintf("rtmp://%v/app/%v", rtmpEndpoint, resp.GetStreamKey()))

	return nil
}

func (b *backend) runOutput(looper *flvlooper.File, outputURL string) {
	ms := rtmp.NewMediaStream(context.Background(), "output")

	go runner.Run(b.logger, looper, ms)

	for {
		errCh := make(chan error)

		// Create a new RTMP pusher
		pusher := rtmppusher.NewPusher(errCh, b.logger)

		// Initiate the connection to the output URL and push the data from the supplied mediastream to the server
		err := pusher.Push(outputURL, ms)

		// We couldn't connect to the output, so wait and retry
		if err != nil {
			pusher.Close()
			b.retryWithError("Error initialising output stream", err)
			continue
		}

		// This will block until the pusher encounters an unrecoverable error
		err = <-errCh

		// If we were intentionally closed then just return, otherwise close the pusher ourselves and spawn a new one
		if pusher.IsClosed() {
			b.logger.Debug("Output closed")
			return
		}

		// Wait and retry
		pusher.Close()
		b.retryWithError("Error in output stream", err)
	}
}

func (b *backend) retryWithError(description string, err error) {
	b.logger.Warnf(description+", retrying in %v %v", retryTime, err)
	time.Sleep(retryTime)
}
