package codec

import (
	"bytes"
	"testing"
)

var (
	sampleAVCD = []byte{0x1, 0x42, 0xc0, 0x20, 0xff, 0xe1, 0x0, 0x22, 0x67, 0x42, 0xc0, 0x20, 0xda, 0x1, 0x40, 0x16, 0xec, 0x4, 0x40, 0x0, 0x0, 0x3, 0x0, 0x40, 0x0, 0x0, 0xf, 0x19, 0xb0, 0x0, 0x3d, 0x9, 0x0, 0x7, 0xa1, 0x39, 0xa8, 0x80, 0x1e, 0x30, 0x65, 0x40, 0x1, 0x0, 0x4, 0x68, 0xce, 0x3c, 0x80}
)

func dupBytes(s []byte) []byte {
	return append([]byte(nil), s...)
}

func TestAVCDecoderConfigGood(t *testing.T) {
	avcd, err := NewAVCDecoderConfig(bytes.NewReader(sampleAVCD))
	if err != nil {
		t.Fatal(err.Error())
	}
	if len(avcd.SpsTable) != 1 {
		t.Fatalf("Expected 1 SPS, found %d", len(avcd.SpsTable))
	}
	if len(avcd.PpsTable) != 1 {
		t.Fatalf("Expected 1 PPS, found %d", len(avcd.PpsTable))
	}
}

func TestAVCDecoderConfigPadded(t *testing.T) {
	padded := append(dupBytes(sampleAVCD), []byte{0xFF, 0xFF, 0xFF, 0xFF}...)
	avcd, err := NewAVCDecoderConfig(bytes.NewReader(padded))
	if err != nil {
		t.Fatal("Failed to parse padded AVCD:", err)
	}
	if len(avcd.SpsTable) != 1 {
		t.Fatalf("Expected 1 SPS, found %d", len(avcd.SpsTable))
	}
	if len(avcd.PpsTable) != 1 {
		t.Fatalf("Expected 1 PPS, found %d", len(avcd.PpsTable))
	}
}

func TestAVCDecoderConfigTruncated(t *testing.T) {
	for i := 0; i < len(sampleAVCD); i++ {
		avcd, err := NewAVCDecoderConfig(bytes.NewReader(sampleAVCD[:i]))
		if err == nil {
			t.Fatalf("Parsed truncated AVCD: bytes %#v avcd %#v err %v", sampleAVCD[:i], avcd, err)
		}
	}
}

func TestAVCDecoderConfigBadVersion(t *testing.T) {
	_, err := NewAVCDecoderConfig(bytes.NewReader([]byte{2}))
	if err != ErrUnknownVersion {
		t.Fatalf("Parsed a misversioned AVCD: err %v", err)
	}
}
