package flv

import (
	"encoding/binary"
	"io"
	"os"
)

type WriteSeekCloseSyncer interface {
	io.WriteSeeker
	io.Closer
	Sync() error
}

type FlvFileWriter struct {
	file              WriteSeekCloseSyncer
	name              string
	headerBuf         []byte
	duration          float64
	firstTimestampSet bool
	firstTimestamp    uint32
}

func CreateFile(name string) (*FlvFileWriter, error) {
	file, err := os.Create(name)
	if err != nil {
		return nil, err
	}
	return WrapFileLike(file, name)
}

func WrapFileLike(fileLike WriteSeekCloseSyncer, name string) (*FlvFileWriter, error) {
	// Write flv header
	if _, err := fileLike.Write(HEADER_BYTES); err != nil {
		fileLike.Close()
		return nil, err
	}

	// Sync to disk
	if err := fileLike.Sync(); err != nil {
		fileLike.Close()
		return nil, err
	}

	return &FlvFileWriter{
		file:      fileLike,
		name:      name,
		headerBuf: make([]byte, 11),
		duration:  0.0,
	}, nil
}

func (flvFile *FlvFileWriter) Close() {
	flvFile.file.Close()
}

// Data with audio header
func (flvFile *FlvFileWriter) WriteAudioTag(data []byte, timestamp uint32) (err error) {
	return flvFile.WriteTag(data, AUDIO_TAG, timestamp)
}

// Data with video header
func (flvFile *FlvFileWriter) WriteVideoTag(data []byte, timestamp uint32) (err error) {
	return flvFile.WriteTag(data, VIDEO_TAG, timestamp)
}

// Write tag
func (flvFile *FlvFileWriter) WriteTag(data []byte, tagType byte, timestamp uint32) (err error) {
	if !flvFile.firstTimestampSet {
		flvFile.firstTimestampSet = true
		flvFile.firstTimestamp = timestamp
	}
	if timestamp > flvFile.firstTimestamp {
		timestamp -= flvFile.firstTimestamp
	} else {
		timestamp = 0
	}
	flvFile.duration = float64(timestamp) / 1000.0
	binary.BigEndian.PutUint32(flvFile.headerBuf[3:7], timestamp)
	flvFile.headerBuf[7] = flvFile.headerBuf[3]
	binary.BigEndian.PutUint32(flvFile.headerBuf[:4], uint32(len(data)))
	flvFile.headerBuf[0] = tagType
	// Write data
	if _, err = flvFile.file.Write(flvFile.headerBuf); err != nil {
		return
	}

	//tmpBuf := make([]byte, 4)
	//// Write tag header
	//if _, err = flvFile.file.Write([]byte{tagType}); err != nil {
	//	return
	//}

	//// Write tag size
	//binary.BigEndian.PutUint32(tmpBuf, uint32(len(data)))
	//if _, err = flvFile.file.Write(tmpBuf[1:]); err != nil {
	//	return
	//}

	//// Write timestamp
	//binary.BigEndian.PutUint32(tmpBuf, timestamp)
	//if _, err = flvFile.file.Write(tmpBuf[1:]); err != nil {
	//	return
	//}
	//if _, err = flvFile.file.Write(tmpBuf[:1]); err != nil {
	//	return
	//}

	//// Write stream ID
	//if _, err = flvFile.file.Write([]byte{0, 0, 0}); err != nil {
	//	return
	//}

	// Write data
	if _, err = flvFile.file.Write(data); err != nil {
		return
	}

	// Write previous tag size
	if err = binary.Write(flvFile.file, binary.BigEndian, uint32(len(data)+11)); err != nil {
		return
	}

	// Sync to disk
	//if err = flvFile.file.Sync(); err != nil {
	//	return
	//}
	return
}

func (flvFile *FlvFileWriter) SetDuration(duration float64) {
	flvFile.duration = duration
}

func (flvFile *FlvFileWriter) Sync() (err error) {
	// Update duration on MetaData
	if _, err = flvFile.file.Seek(DURATION_OFFSET, 0); err != nil {
		return
	}
	if err = binary.Write(flvFile.file, binary.BigEndian, flvFile.duration); err != nil {
		return
	}
	if _, err = flvFile.file.Seek(0, 2); err != nil {
		return
	}

	err = flvFile.file.Sync()
	return
}

func (flvFile *FlvFileWriter) FilePath() string {
	return flvFile.name
}
