package player

import (
	"fmt"
	"net"
	"net/url"
	"reflect"
	"strings"
	"sync"

	"code.justin.tv/event-engineering/gortmp/pkg/amf"
	"code.justin.tv/event-engineering/gortmp/pkg/log"
	"code.justin.tv/event-engineering/gortmp/pkg/rtmp"
	goctx "context"
)

type TransactionHandler func(rtmp.Message) error
type TransactionMap struct {
	mu    sync.Mutex
	m     map[uint32]TransactionHandler
	txnid uint32
}

type MediaPlayerHandler interface {
	Handle(goctx.Context, rtmp.Receiver, rtmp.Message) error
}

func NewTransactionMap() *TransactionMap {
	return &TransactionMap{
		m: make(map[uint32]TransactionHandler),
	}
}

func (tm *TransactionMap) New(h TransactionHandler) uint32 {
	tm.mu.Lock()
	defer tm.mu.Unlock()

	id := tm.txnid
	tm.txnid++
	tm.m[id] = h

	return id
}

func (tm *TransactionMap) Handle(id uint32, msg rtmp.Message) error {
	if h, ok := tm.m[id]; ok {
		delete(tm.m, id)
		return h(msg)
	}
	return nil
}

type RtmpPlayer struct {
	URL  *url.URL
	Conn rtmp.BasicConn
	tm   *TransactionMap
}

func (p *RtmpPlayer) ConnectResult(msg rtmp.Message) error {
	cmd := rtmp.CreateStreamCommand{TransactionID: p.tm.New(p.CreateStreamResult)}
	return p.Send(cmd)
}

func (p *RtmpPlayer) CreateStreamResult(msg rtmp.Message) error {
	result, ok := msg.(rtmp.ResultCommand)
	if !ok {
		return fmt.Errorf("Got invalid result command for CreateStream: %#v", msg)
	}

	log.Infof(goctx.Background(), "CreateStreamResult: %#v", result)

	log.Infof(goctx.Background(), "TypeOf(result.Info)=%s", reflect.TypeOf(result.Info).Name())

	streamID, ok := result.Info.(float64)
	if !ok {
		return fmt.Errorf("Invalid result commend for CreateStream: %#v", result)
	}

	path := strings.Split(p.URL.Path, "/")
	if len(path) == 0 {
		return fmt.Errorf("Unable to parse stream name: %s", p.URL.Path)
	}

	cmd := rtmp.PlayCommand{
		StreamID:      uint32(streamID),
		TransactionID: p.tm.New(p.PlayResult),
		Name:          path[len(path)-1],
	}

	return p.Send(cmd)
}

func (p *RtmpPlayer) PlayResult(msg rtmp.Message) error {
	log.Infof(goctx.Background(), "Beginning playback for %s", p.URL)
	return nil
}

func (p *RtmpPlayer) Prologue() error {
	// first just send the connect command
	tcURL := *p.URL
	tcURL.Path = "app"
	connectCmd := rtmp.ConnectCommand{
		TransactionID: p.tm.New(p.ConnectResult),
		Properties: amf.Object{
			"app":           "app",
			"flashVer":      "LNX 9,0,124,2",
			"tcUrl":         tcURL.String(),
			"fpad":          false,
			"capabilities":  15,
			"audioCodecs":   4071,
			"videoCodecs":   252,
			"videoFunction": 1,
		},
	}

	log.Infof(goctx.Background(), "Writing connect msg: %+v", connectCmd)
	if err := p.Send(connectCmd); err != nil {
		return err
	}

	if err := p.Send(rtmp.SetChunkSizeMessage{Size: 4096}); err != nil {
		return err
	}
	return nil
}

func (p *RtmpPlayer) PlayRTMP(h MediaPlayerHandler) error {
	if err := p.Prologue(); err != nil {
		return err
	}

	for {
		raw, err := p.Conn.Read()
		if err != nil {
			return err
		}

		msg, err := p.parseMessage(raw)
		if err != nil {
			return err
		}
		if err := h.Handle(goctx.Background(), p, msg); err != nil {
			return err
		}
	}
}

func (p *RtmpPlayer) Handle(msg rtmp.Message) error {
	switch msg := msg.(type) {
	case rtmp.ResultCommand:
		return p.tm.Handle(msg.TransactionID, msg)
	case rtmp.ErrorCommand:
		return p.tm.Handle(msg.TransactionID, msg)
	}
	return nil
}

func (p *RtmpPlayer) parseMessage(raw *rtmp.RawMessage) (rtmp.Message, error) {
	if raw.ChunkStreamID == rtmp.CS_ID_PROTOCOL_CONTROL {
		var msg rtmp.Message
		var err error

		if raw.Type == rtmp.USER_CONTROL_MESSAGE {
			msg, err = rtmp.ParseEvent(raw)
		} else {
			msg, err = rtmp.ParseControlMessage(raw)
		}

		return msg, err
	}

	switch raw.Type {
	case rtmp.COMMAND_AMF0:
		fallthrough
	case rtmp.COMMAND_AMF3:
		return rtmp.ParseCommand(raw)
	case rtmp.USER_CONTROL_MESSAGE:
		return rtmp.ParseEvent(raw)
	default:
		return raw, nil
	}
}

func (p *RtmpPlayer) Send(msg rtmp.Message) error {
	raw, err := msg.RawMessage()
	if err != nil {
		return err
	}

	if err := p.Conn.Write(raw); err != nil {
		return err
	}

	return p.Conn.Flush()
}

func dial(host string) (net.Conn, error) {
	hostPort := strings.Split(host, ":")
	var port string
	if len(hostPort) > 1 {
		host, port = hostPort[0], hostPort[1]
	} else {
		port = "1935"
	}

	host = net.JoinHostPort(host, port)

	return net.Dial("tcp", host)
}

func NewRtmpPlayer(rtmpURL string) (*RtmpPlayer, error) {
	parsed, err := url.Parse(rtmpURL)
	if err != nil {
		return nil, fmt.Errorf("Invalid rtmp url: %s", err)
	}

	conn, err := dial(parsed.Host)
	if err != nil {
		return nil, err
	}

	log.Infof(goctx.Background(), "Connected to %s, performing handshake", parsed.Host)
	if _, err := rtmp.Handshake(goctx.Background(), conn); err != nil {
		return nil, fmt.Errorf("Handshake failed: %s", err)
	}

	player := &RtmpPlayer{
		URL:  parsed,
		Conn: rtmp.NewBasicConn(conn),
		tm:   NewTransactionMap(),
	}
	return player, nil
}
