package main

import (
	"code.justin.tv/event-engineering/rtmp/pkg/rtmptail"
	"fmt"
	"github.com/spf13/cobra"
	"os"
	"os/signal"
	"syscall"
)

func init() {
	rootCmd.AddCommand(rtmpTailCommand)
	rtmpTailCommand.PersistentFlags().StringP("input", "i", "", "Input RTMP URL")
	rtmpTailCommand.PersistentFlags().StringP("output", "o", "", "Output file")
	rtmpTailCommand.PersistentFlags().IntP("seconds", "s", 30, "Number of seconds of data to keep")
}

var rtmpTailCommand = &cobra.Command{
	Use:   "tail",
	Short: "Keep the last -s seconds of RTMP data in a buffer and dump it to file on interrupt",
	Run: func(cmd *cobra.Command, args []string) {
		input, err := cmd.Flags().GetString("input")
		if err != nil {
			fmt.Println("Unable to parse input ", err)
			os.Exit(1)
		}

		output, err := cmd.Flags().GetString("output")
		if err != nil {
			fmt.Println("Unable to parse input ", err)
			os.Exit(1)
		}

		seconds, err := cmd.Flags().GetInt("seconds")
		if err != nil {
			fmt.Println("Unable to parse input ", err)
			os.Exit(1)
		}

		if input == "" {
			fmt.Println("No input specified, use -i")
			os.Exit(1)
		}

		if output == "" {
			fmt.Println("No output specified, use -o")
			os.Exit(1)
		}

		if seconds == 0 {
			fmt.Println("No duration specified, use -s")
			os.Exit(1)
		}

		errCh, ender, finalised := rtmptail.Tail(input, output, seconds, logger)
		fmt.Printf("Buffering %vs of data\n", seconds)

		osSig := make(chan os.Signal, 1)
		signal.Notify(osSig, os.Interrupt, syscall.SIGTERM, syscall.SIGQUIT)

		go func() {
			err := <-errCh
			fmt.Printf("Error dumping RTMP %v\n", err)
			os.Exit(1)
		}()

		sig := <-osSig
		fmt.Printf("Finalising output after receiving signal %v\n", sig)
		close(ender)

		<-finalised

		fmt.Printf("Saved output to %v\n", output)
	},
}
