package conn

import (
	"net"
	"time"
)

// Conn just wraps net.Conn but with a specified Write idle timeout
type Conn struct {
	conn             net.Conn
	readIdleTimeout  time.Duration
	writeIdleTimeout time.Duration
}

// New creates a new OutputConn wrapping the supplied net.Conn with the suppied idleTimeouts
func New(conn net.Conn, readIdleTimeout time.Duration, writeIdleTimeout time.Duration) net.Conn {
	ic := &Conn{
		conn:             conn,
		readIdleTimeout:  readIdleTimeout,
		writeIdleTimeout: writeIdleTimeout,
	}

	return ic
}

// Read calls net.Conn.Read() after calling net.Conn.SetReadDeadline() with the timeout supplied to OutputConn
func (ic *Conn) Read(b []byte) (n int, err error) {
	if ic.readIdleTimeout > 0 {
		deadlineErr := ic.conn.SetReadDeadline(time.Now().Add(ic.readIdleTimeout))
		if deadlineErr != nil {
			return 0, err
		}
	}

	return ic.conn.Read(b)
}

// Write calls net.Conn.Write() after calling net.Conn.SetWriteDeadline() with the timeout supplied to OutputConn
func (ic *Conn) Write(b []byte) (n int, err error) {
	if ic.writeIdleTimeout > 0 {
		deadlineErr := ic.conn.SetWriteDeadline(time.Now().Add(ic.writeIdleTimeout))
		if deadlineErr != nil {
			return 0, err
		}
	}

	return ic.conn.Write(b)
}

// Close calls net.Conn.Close()
func (ic *Conn) Close() error {
	return ic.conn.Close()
}

// LocalAddr calls net.Conn.LocalAddr()
func (ic *Conn) LocalAddr() net.Addr {
	return ic.conn.LocalAddr()
}

// RemoteAddr calls net.Conn.RemoteAddr()
func (ic *Conn) RemoteAddr() net.Addr {
	return ic.conn.RemoteAddr()
}

// SetDeadline calls net.Conn.SetDeadline()
func (ic *Conn) SetDeadline(t time.Time) error {
	return ic.conn.SetDeadline(t)
}

// SetReadDeadline calls net.Conn.SetReadDeadline()
func (ic *Conn) SetReadDeadline(t time.Time) error {
	return ic.conn.SetReadDeadline(t)
}

// SetWriteDeadline calls net.Conn.SetWriteDeadline()
func (ic *Conn) SetWriteDeadline(t time.Time) error {
	return ic.conn.SetWriteDeadline(t)
}
