package rtmpdump

import (
	"context"
	"github.com/pkg/errors"

	logging "code.justin.tv/event-engineering/golibs/pkg/logging"
	player "code.justin.tv/event-engineering/rtmp/pkg/player"
	flv "code.justin.tv/video/goflv"
	rtmplog "code.justin.tv/event-engineering/gortmp/pkg/log"
	gortmp "code.justin.tv/event-engineering/gortmp/pkg/rtmp"
)

// Dump will dump the rtmp from input to the output
func Dump(url string, outputFile string, logger logging.Logger) (errCh chan error, ender chan interface{}, finalised chan interface{}) {
	rtmplog.SetLogLevel(rtmplog.LogError)
	player := player.NewPlayer(errCh, logger)

	ctx := context.Background()
	ms := gortmp.NewMediaStream(ctx, "dumpthis")
	err := player.Play(url, ms)
	if err != nil {
		errCh <- errors.Wrap(err, "Could not connect to stream")
		return
	}

	ender = make(chan interface{}, 1)
	finalised = make(chan interface{}, 1)
	errCh = make(chan error)

	// Grab rtmp messages from the connection and buffer them
	go dumpLoop(ms, player, outputFile, errCh, ender, finalised)

	return errCh, ender, finalised
}

func dumpLoop(ms gortmp.MediaStream, player *player.Player, outputFile string, errCh chan error, ender chan (interface{}), finalised chan (interface{})) {
	var initialTimestamp uint32

	// Create the output file here so we can bomb out if we don't have permission rather than waiting until
	// finalisation just to find out you can't even write to the file
	flvFile, err := flv.CreateFile(outputFile)

	defer flvFile.Close()

	if err != nil {
		errCh <- errors.Wrap(err, "Could not open output file")
		return
	}

	tags, err := ms.Subscribe()
	if err != nil {
		errCh <- errors.Wrap(err, "Could not subscribe to media stream")
		return
	}

	for {
		select {
		case <-ender:
			ms.Unsubscribe(tags)
			player.Close()
			flvFile.Sync()

			// Signal that we're done
			close(finalised)

			return
		case tag := <-tags:
			if tag.Timestamp > 0 {
				if initialTimestamp == 0 {
					initialTimestamp = tag.Timestamp
				}

				tag.Timestamp = tag.Timestamp - initialTimestamp
			}

			err = flvFile.WriteTag(tag.Bytes, tag.Type, tag.Timestamp)
			if err != nil {
				errCh <- errors.Wrap(err, "Error writing flv tag")
				return
			}
		}
	}
}
