package main

import (
	"bytes"
	"flag"
	"log"

	"code.justin.tv/video/gortmp/pkg/amf"
	rtmplog "code.justin.tv/video/gortmp/pkg/log"
	"code.justin.tv/video/gortmp/pkg/player"
	"code.justin.tv/video/gortmp/pkg/rtmp"
	goctx "context"
)

var rtmpURL = flag.String("rtmp", "", "rtmp connect url")
var test = flag.Bool("test", false, "just check if the stream is up")

type playerHandler struct{}

func unpackDataTag(r *bytes.Buffer) ([]interface{}, error) {
	if r == nil {
		return nil, nil
	}

	var objs []interface{}
	for r.Len() > 0 {
		obj, err := amf.ReadValue(r)
		if err != nil {
			return objs, err
		}
		objs = append(objs, obj)
	}
	return objs, nil
}

func (h *playerHandler) Handle(ctx goctx.Context, r rtmp.Receiver, msg rtmp.Message) error {
	switch msg := msg.(type) {
	case *rtmp.RawMessage:
		var objs []interface{}
		if msg.Type == 18 {
			objs, _ = unpackDataTag(msg.Data)
		}
		rtmplog.Infof(ctx, "RawMessage: {ChunkStreamID: %d StreamID: %d Timestamp: %d Type: %d} - %+v",
			msg.ChunkStreamID, msg.StreamID, msg.Timestamp, msg.Type, objs)
	case rtmp.Command:
		rtmplog.Infof(ctx, "Command: %#v", msg)
	default:
		rtmplog.Infof(ctx, "Msg: %#v", msg)
	}
	return r.Handle(msg)
}

func testPlay(p *player.RtmpPlayer) bool {
	err := p.PlayRTMP(&playerHandler{})
	return err == nil
}

func main() {
	rtmplog.SetLogLevel(rtmplog.LogDebug)
	log.SetFlags(log.Lmicroseconds)
	flag.Parse()
	rtmpPlayer, err := player.NewRtmpPlayer(*rtmpURL)
	if err != nil {
		log.Fatalf("Error connecting stream: %s", err)
	}
	if *test {
		ok := testPlay(rtmpPlayer)
		rtmpPlayer.Conn.Close()
		if ok {
			log.Println("Test result: OK")
		} else {
			log.Println("Test result: NOT OK")
		}
	} else {
		if err = rtmpPlayer.PlayRTMP(&playerHandler{}); err != nil {
			log.Fatalf("Error playing stream: %s", err)
		}
	}
}
