package main

import (
	rtmpctx "code.justin.tv/video/gortmp/pkg/context"
	rtmplog "code.justin.tv/video/gortmp/pkg/log"
	"code.justin.tv/video/gortmp/pkg/rtmp"
	"container/list"
	goctx "context"
	"flag"
	"fmt"
	golog "log"
	"net"
	"net/http"
	_ "net/http/pprof"
	"os"
	"time"
)

var (
	listenAddr = flag.String("listen", ":1935", "The address to bind to")
	logFlag    = flag.String("log-level", "trace", "Log level")
)

func TrackStarvation(ctx goctx.Context, ms rtmp.MediaStream) {
	log := rtmplog.FromContext(ctx, "trackstarvation")
	log.Infof("start")
	defer log.Infof("end")

	ch, err := ms.Subscribe()
	if err != nil {
		log.Errorf("stream subscribe failed")
		return
	}

	defer ms.Unsubscribe(ch)
	ticker := time.NewTicker(5 * time.Second)
	defer ticker.Stop()

	var curTs, lastTs int64
	var lastClock time.Time
	history := list.New()

	for {
		select {
		case tag, ok := <-ch:
			if ok {
				if tag.Type == rtmp.VIDEO_TYPE {
					curTs = int64(tag.Timestamp)
					log.Tracef("tag.Timestamp=%d", tag.Timestamp)
				}
			} else {
				return
			}
		case <-ticker.C:
			if lastClock.IsZero() {
				lastTs = curTs
				lastClock = time.Now()
			} else {
				streamDelta := (curTs - lastTs)
				clockDelta := time.Since(lastClock).Nanoseconds() / 1000000
				log.Debugf("streamDelta=%d,clockDelta=%d", streamDelta, clockDelta)
				lastTs = curTs
				lastClock = time.Now()
				history.PushFront(clockDelta - streamDelta)

				if history.Len() > 25 {
					history.Remove(history.Back())
				}

				var sum int64

				for e := history.Front(); e != nil; e = e.Next() {
					sum += e.Value.(int64)
				}

				// this mimics the behavior of the wowza starvation tracker
				log.Infof("sum = %d, len = %d", sum, history.Len())
				log.Infof("starved = %t", float64(sum)/float64(clockDelta)/float64(history.Len()) > 0.05)
				log.Infof("starvation = %d", sum/int64(history.Len()))
			}
		}
	}
}

func main() {
	flag.Parse()
	logger := golog.New(os.Stdout, "", golog.Lshortfile)
	logLevel := rtmplog.ParseLogLevel(*logFlag)
	rtmplog.SetLogLevel(logLevel)

	go func() {
		err := http.ListenAndServe(":6060", nil)
		if err != nil {
			logger.Fatalf("pprof listener failed: %s", err)
		}
	}()

	server := rtmp.NewServer(rtmp.ServerConfig{
		Handler: &ProcessSpawner{},
		Logger:  logger,
	})

	logger.Printf("ListenAndServe(%s)", *listenAddr)
	if err := server.ListenAndServe(*listenAddr); err != nil {
		fmt.Printf("Server error: %s\n", err)
	} else {
		fmt.Printf("Server exited cleanly: %s\n", err)
	}
}

type handler struct {
}

func (h *handler) OnMediaStreamCreated(goctx.Context, rtmp.MediaStream) {
}

func (h *handler) OnMediaStreamDestroyed(goctx.Context, rtmp.MediaStream) {
}

func (h *handler) Handle(ctx goctx.Context, r rtmp.Receiver, msg rtmp.Message) error {
	log := rtmplog.FromContext(ctx, "rtmpserver.handler")
	if _, ok := msg.(*rtmp.RawMessage); !ok {
		log.Infof("%#v", msg)
	} else {
		raw, err := msg.RawMessage()
		if err != nil {
			return err
		}

		if raw.Type != rtmp.VIDEO_TYPE && raw.Type != rtmp.AUDIO_TYPE {
			log.Infof("%#v", raw)
		}
	}
	return r.Handle(msg)
}

type ProcessSpawner struct{}

func (ps *ProcessSpawner) ServeRTMP(ctx goctx.Context, conn net.Conn) error {
	// start separate goingest process
	log := rtmplog.FromContext(ctx, "rtmpserver.ProcessSpawner")
	tcpConn, ok := conn.(*net.TCPConn)
	if ok {
		if fd, err := tcpConn.File(); err != nil {
			return err
		} else {

			attrs := &os.ProcAttr{Files: []*os.File{os.Stdin, os.Stdout, os.Stderr, fd}}
			args := []string{"rtmpworker"}
			args = append(args, os.Args[1:]...)
			p, err := os.StartProcess("rtmpworker", args, attrs)
			if err != nil {
				return err
			}
			conn.Close()

			log.Infof("New process created with PID: %d", p.Pid)

			//return NewProcess(ctx, fd)
		}
	}
	return nil

}

func NewProcess(ctx goctx.Context, fd *os.File) error {
	conn, err := net.FileConn(fd)
	if err != nil {
		return err
	}
	logger, ok := rtmpctx.GetLogger(ctx)
	if !ok {
		logger = golog.New(os.Stdout, "", golog.Lshortfile)
	}
	ms := rtmp.NewMediaServer(&handler{})
	go ms.ServeRTMP(ctx, conn)
	server := rtmp.NewServer(rtmp.ServerConfig{
		Handler: ms,
		Logger:  logger,
	})
	if err := server.ListenAndServe(":0"); err != nil {
		fmt.Printf("Server error: %s\n", err)
	} else {
		fmt.Printf("Server exited cleanly: %s\n", err)
	}
	return nil
}
