package main

import (
	rtmplog "code.justin.tv/video/gortmp/pkg/log"
	"code.justin.tv/video/gortmp/pkg/rtmp"
	goctx "context"
	"flag"
	"fmt"
	golog "log"
	"net"
	"os"
	"strings"
)

var (
	listenAddr = flag.String("listen", ":1935", "The address to bind to")
	logFlag    = flag.String("log-level", "trace", "Log level")
)

func main() {
	flag.Parse()
	fd := os.NewFile(3, "")
	conn, err := net.FileConn(fd)
	if err != nil {
		fmt.Printf("Could not get net.Conn from fd 3")
		return
	}
	if err := Worker(conn); err != nil {
		fmt.Printf("Worker error: %s", err)
	}
	fd.Close()
}

func Worker(conn net.Conn) error {
	logger := golog.New(os.Stdout, "", golog.Lshortfile)

	rtmpEventHandler := &handler{}
	ms := rtmp.NewMediaServer(rtmpEventHandler)
	rtmpEventHandler.mediaServer = ms

	go ms.ServeRTMP(goctx.TODO(), conn)

	server := rtmp.NewServer(rtmp.ServerConfig{
		Handler: ms,
		Logger:  logger,
	})

	if err := server.ListenAndServe(":0"); err != nil {
		fmt.Printf("Server error: %s\n", err)
		return err
	} else {
		fmt.Println("Server exited cleanly")
	}

	return nil
}

type handler struct {
	mediaServer *rtmp.MediaServer
}

func (h *handler) OnMediaStreamCreated(ctx goctx.Context, r rtmp.MediaStream) {
	fmt.Println("OnMediaStreamCreated")

	if strings.HasSuffix(r.Name(), "_copy") {
		return
	}

	newCtx := goctx.TODO()
	copyMediaStream := rtmp.NewMediaStream(newCtx, r.Name()+"_copy")
	h.mediaServer.AddStream(newCtx, copyMediaStream)
	tags, err := r.Subscribe()

	if err != nil {
		fmt.Printf("Error subscribing wtf: %s\n", err)
		return
	}

	go func(instream chan *rtmp.FlvTag, ms rtmp.MediaStream) {
		for {
			curTag, gotMsg := <-instream
			if !gotMsg {
				break
			}
			ms.Publish(curTag)
		}
	}(tags, copyMediaStream)
}

func (h *handler) OnMediaStreamDestroyed(goctx.Context, rtmp.MediaStream) {
}

func (h *handler) Handle(ctx goctx.Context, r rtmp.Receiver, msg rtmp.Message) error {
	logLevel := rtmplog.ParseLogLevel(*logFlag)
	rtmplog.SetLogLevel(logLevel)
	log := rtmplog.FromContext(ctx, "rtmpserver.handler")
	if _, ok := msg.(*rtmp.RawMessage); !ok {
		log.Infof("Handle: %#v", msg)
	} else {
		raw, err := msg.RawMessage()
		if err != nil {
			return err
		}

		if raw.Type != rtmp.VIDEO_TYPE && raw.Type != rtmp.AUDIO_TYPE {
			log.Infof("%#v", raw)
		}
	}
	return r.Handle(msg)
}
