package flv

import (
	"io"
	"testing"

	"github.com/orcaman/writerseeker"
)

type nopCloseSyncer struct {
	io.WriteSeeker
}

func (nopCloseSyncer) Close() error {
	return nil
}

func (nopCloseSyncer) Sync() error {
	return nil
}

func checkTag(t *testing.T, r *FlvFileReader, expectedTs uint32, expectedType uint8, expectedDataLen int) {
	hdr, data, err := r.ReadTag()
	if err != nil {
		t.Fatal(err)
	}
	if hdr.Timestamp != expectedTs {
		t.Fatalf("timestamp corrupted %d (expected %d)", hdr.Timestamp, expectedTs)
	}
	if hdr.TagType != expectedType {
		t.Fatalf("tag type corrupted: %d (expected %d)", hdr.TagType, expectedType)
	}
	if len(data) != expectedDataLen {
		t.Fatalf("tag length corrupted: %d (expected %d)", len(data), expectedDataLen)
	}
}

func TestBigTimestamps(t *testing.T) {
	wb := &writerseeker.WriterSeeker{}
	w, err := WrapFileLike(nopCloseSyncer{wb}, "")
	if err != nil {
		t.Fatal(err)
	}
	// Avoid timestamps with the high bit set, they are underspecified in FLV file format
	const maxTs uint32 = 1<<28 - 1
	// This is not a realistic value. A fairly large prime is used here so that this runs
	// fast and tends to put unique values in each byte.
	const tsStep = uint32(7919)
	const tagType = AUDIO_TAG

	for ts := uint32(0); ts < maxTs; ts += tsStep {
		err := w.WriteTag(nil, tagType, ts)
		if err != nil {
			t.Fatal(err)
		}
	}
	w.Sync()
	w.Close()
	rb := wb.BytesReader()
	r, err := WrapInMemoryFile(rb, "")
	if err != nil {
		t.Fatal(err)
	}

	// Get implicit metadata tag
	checkTag(t, r, 0, SCRIPT_DATA_TAG, 40)

	for ts := uint32(0); ts < maxTs; ts += tsStep {
		checkTag(t, r, ts, AUDIO_TAG, 0)
	}
}
