package log

import (
	"fmt"
	"log"
	"strings"

	rtmpctx "code.justin.tv/video/gortmp/pkg/context"
	"context"
)

type LogLevel int

const (
	LogError LogLevel = iota
	LogWarn
	LogInfo
	LogDebug
	LogTrace
)

var logLevel LogLevel = LogInfo
var logSep = " - "

func (l LogLevel) String() string {
	switch l {
	case LogError:
		return "error"
	case LogWarn:
		return "warn"
	case LogInfo:
		return "info"
	case LogDebug:
		return "debug"
	case LogTrace:
		return "trace"
	default:
		return "undefined"
	}
}

func ParseLogLevel(l string) LogLevel {
	switch l {
	case "error":
		return LogError
	case "warn":
		return LogWarn
	case "info":
		return LogInfo
	case "debug":
		return LogDebug
	case "trace":
		return LogTrace
	default:
		return LogInfo
	}
}

func SetLogLevel(level LogLevel) {
	logLevel = level
}

type Logger interface {
	Errorf(string, ...interface{})
	Warnf(string, ...interface{})
	Infof(string, ...interface{})
	Debugf(string, ...interface{})
	Tracef(string, ...interface{})
	Output(LogLevel, string, ...interface{})
}

type logger struct {
	tags   []string
	logger *log.Logger
}

func FromContext(ctx context.Context, tags ...string) Logger {
	l := &logger{
		tags: append(logTags(ctx), tags...),
	}

	if logger, ok := rtmpctx.GetLogger(ctx); ok {
		l.logger = logger
	}

	return l
}

func logTags(ctx context.Context) []string {
	tags := []string{}

	if addr, ok := rtmpctx.GetListenAddr(ctx); ok {
		tags = append(tags, addr.String())
	}

	if addr, ok := rtmpctx.GetRemoteAddr(ctx); ok {
		tags = append(tags, addr.String())
	}

	if status, ok := rtmpctx.GetConnStatus(ctx); ok {
		tags = append(tags, status)
	}

	if name, ok := rtmpctx.GetStreamName(ctx); ok {
		tags = append(tags, name)
	}

	return tags
}

func (l *logger) Output(level LogLevel, format string, v ...interface{}) {
	// ignore log messages above the current level
	if logLevel < level {
		return
	}

	tags := append([]string{level.String()}, append(l.tags, format)...)
	joined := strings.Join(tags, logSep)

	if l.logger != nil {
		l.logger.Output(3, fmt.Sprintf(joined, v...))
	} else {
		log.Printf(joined, v...)
	}
}

func (l *logger) Errorf(format string, v ...interface{}) {
	l.Output(LogError, format, v...)
}

func (l *logger) Warnf(format string, v ...interface{}) {
	l.Output(LogWarn, format, v...)
}

func (l *logger) Infof(format string, v ...interface{}) {
	l.Output(LogInfo, format, v...)
}

func (l *logger) Debugf(format string, v ...interface{}) {
	l.Output(LogDebug, format, v...)
}

func (l *logger) Tracef(format string, v ...interface{}) {
	l.Output(LogTrace, format, v...)
}

func Errorf(ctx context.Context, format string, v ...interface{}) {
	FromContext(ctx).Output(LogError, format, v...)
}

func Warnf(ctx context.Context, format string, v ...interface{}) {
	FromContext(ctx).Output(LogWarn, format, v...)
}

func Infof(ctx context.Context, format string, v ...interface{}) {
	FromContext(ctx).Output(LogInfo, format, v...)
}

func Debugf(ctx context.Context, format string, v ...interface{}) {
	FromContext(ctx).Output(LogDebug, format, v...)
}

func Tracef(ctx context.Context, format string, v ...interface{}) {
	FromContext(ctx).Output(LogTrace, format, v...)
}
