package rtmp

import (
	"bytes"
	"encoding/binary"
	"io"
	"math/rand"
	"testing"
	"time"
)

type TestHeader struct {
	ChunkStreamID uint32
	Timestamp     uint32
	Type          uint8
	Length        uint32
	StreamID      uint32
}

type TestChunk struct {
	Fmt uint8
	TestHeader
	Payload []byte
}

type TestMessage struct {
	TestHeader
	Payload []byte
}

func init() {
	rand.Seed(time.Now().Unix())
}

func (tc *TestChunk) Data() []byte {
	headerLen := tc.HeaderLen()
	data := make([]byte, headerLen+uint32(len(tc.Payload)))
	cur := data[:]

	// Base header
	if tc.ChunkStreamID <= 63 {
		data[0] = tc.Fmt<<6 | uint8(tc.ChunkStreamID)
		cur = data[1:]
	} else if tc.ChunkStreamID <= 319 {
		data[0] = tc.Fmt << 6
		data[1] = uint8(tc.ChunkStreamID - 64)
		cur = data[2:]
	} else {
		data[0] = tc.Fmt<<6 | 1
		data[1] = uint8((tc.ChunkStreamID - 64) & 0xFF)
		data[2] = uint8((tc.ChunkStreamID - 64) >> 8)
		cur = data[3:]
	}

	// Header
	switch tc.Fmt {
	case HEADER_FMT_FULL:
		binary.LittleEndian.PutUint32(cur[7:], tc.StreamID)
		fallthrough
	case HEADER_FMT_SAME_STREAM:
		PutUint24(cur[3:], tc.Length)
		cur[6] = tc.Type
		fallthrough
	case HEADER_FMT_SAME_LENGTH_AND_STREAM:
		if tc.Timestamp >= 0xffffff {
			PutUint24(cur, 0xffffff)
		} else {
			PutUint24(cur, tc.Timestamp)
		}
		fallthrough
	case HEADER_FMT_CONTINUATION:
		// nothing
	}

	// Extended timestamp
	if tc.Timestamp >= 0xffffff {
		binary.BigEndian.PutUint32(data[headerLen-4:], tc.Timestamp)
	}

	copy(data[headerLen:], tc.Payload)

	return data
}

func (tc *TestChunk) HeaderLen() uint32 {
	var length uint32
	if tc.ChunkStreamID <= 63 {
		length++
	} else if tc.ChunkStreamID <= 319 {
		length += 2
	} else {
		length += 3
	}

	switch tc.Fmt {
	case HEADER_FMT_FULL:
		// uint24 timestamp
		// uint24 message length
		// uint8  message type
		// uint32 stream id
		length += 11
	case HEADER_FMT_SAME_STREAM:
		// uint24 timestamp
		// uint24 message length
		// uint8  message type
		length += 7
	case HEADER_FMT_SAME_LENGTH_AND_STREAM:
		// uint24 timestamp
		length += 3
	case HEADER_FMT_CONTINUATION:
		// nothing
	default:
		panic("invalid header format")
	}

	if tc.Timestamp > 0xffffff {
		// uint32 extended timestamp
		length += 4
	}

	return length
}

func writeChunks(w io.Writer, chunks []*TestChunk) {
	for _, chunk := range chunks {
		_, err := w.Write(chunk.Data())
		if err != nil {
			panic(err)
		}
	}
}

func testChunkStream(chunks []*TestChunk, messages []*TestMessage, t *testing.T) {
	buf := &bytes.Buffer{}
	writeChunks(buf, chunks)
	csr := NewChunkStreamReader(buf)

	t.Logf("Running test stream: %v", buf.Bytes())

	for _, message := range messages {
		msg, err := csr.Read()
		if err != nil {
			t.Fatalf("Failed to read chunk stream: %s", err)
		}

		if testing.Verbose() {
			t.Logf("csr.Read returned: %#v", msg)
		}

		if msg.ChunkStreamID != message.ChunkStreamID {
			t.Fatalf("csid mismatch: %d != %d", msg.ChunkStreamID, message.ChunkStreamID)
		}

		if msg.StreamID != message.StreamID {
			t.Fatalf("stream id mismatch: %d != %d", msg.StreamID, message.StreamID)
		}

		if msg.Type != message.Type {
			t.Fatalf("message type mismatch: %d != %d", msg.Type, message.Type)
		}

		if msg.Timestamp != message.Timestamp {
			t.Fatalf("message timestamp mismatch: %d != %d", msg.Timestamp, message.Timestamp)
		}

		if uint32(msg.Data.Len()) != message.Length {
			t.Fatalf("payload length mismatch")
		}

		if bytes.Compare(msg.Data.Bytes(), message.Payload) != 0 {
			t.Errorf("%v != %v", msg.Data.Bytes(), message.Payload)
			t.Fatalf("payload mismatch")
		}
	}

	if buf.Len() > 0 {
		t.Fatalf("unread data in message stream")
	}
}

func TestOneByteBaseHeader(t *testing.T) {
	chunks := []*TestChunk{
		&TestChunk{
			Fmt: HEADER_FMT_FULL,
			TestHeader: TestHeader{
				Timestamp:     rand.Uint32() & 0xfffff,
				ChunkStreamID: 24, // csid between 2 and 63
				Length:        4,
			},
			Payload: randomPayload(4),
		},
	}

	messages := []*TestMessage{
		&TestMessage{
			TestHeader: chunks[0].TestHeader,
			Payload:    chunks[0].Payload,
		},
	}

	testChunkStream(chunks, messages, t)
}

func TestTwoByteBaseHeader(t *testing.T) {
	chunks := []*TestChunk{
		&TestChunk{
			Fmt: HEADER_FMT_FULL,
			TestHeader: TestHeader{
				Timestamp:     rand.Uint32() & 0xfffff,
				ChunkStreamID: 243, // csid between 64 and 319
				Length:        4,
			},
			Payload: randomPayload(4),
		},
	}

	messages := []*TestMessage{
		&TestMessage{
			TestHeader: chunks[0].TestHeader,
			Payload:    chunks[0].Payload,
		},
	}

	testChunkStream(chunks, messages, t)
}

func TestThreeByteBaseHeader(t *testing.T) {
	chunks := []*TestChunk{
		&TestChunk{
			Fmt: HEADER_FMT_FULL,
			TestHeader: TestHeader{
				Timestamp:     rand.Uint32() & 0xfffff,
				ChunkStreamID: 4000, // csid between 320 and uint16 max
				Length:        4,
			},
			Payload: randomPayload(4),
		},
	}

	messages := []*TestMessage{
		&TestMessage{
			TestHeader: chunks[0].TestHeader,
			Payload:    chunks[0].Payload,
		},
	}

	testChunkStream(chunks, messages, t)
}

func TestFullChunk(t *testing.T) {
	chunks := []*TestChunk{
		&TestChunk{
			Fmt: HEADER_FMT_FULL,
			TestHeader: TestHeader{
				Timestamp:     rand.Uint32() & 0xfffff,
				ChunkStreamID: 2,
				Type:          100,
				Length:        4,
				StreamID:      rand.Uint32(),
			},
			Payload: randomPayload(4),
		}, &TestChunk{
			Fmt: HEADER_FMT_FULL,
			TestHeader: TestHeader{
				Timestamp:     rand.Uint32() & 0xfffff,
				ChunkStreamID: 934,
				Type:          101,
				Length:        4,
				StreamID:      rand.Uint32(),
			},
			Payload: randomPayload(4),
		}, &TestChunk{
			Fmt: HEADER_FMT_FULL,
			TestHeader: TestHeader{
				Timestamp:     rand.Uint32() & 0xfffff,
				ChunkStreamID: 3,
				Type:          102,
				Length:        4,
				StreamID:      rand.Uint32(),
			},
			Payload: randomPayload(4),
		}, &TestChunk{
			Fmt: HEADER_FMT_FULL,
			TestHeader: TestHeader{
				Timestamp:     rand.Uint32() & 0xfffff,
				ChunkStreamID: 124,
				Type:          103,
				Length:        4,
				StreamID:      rand.Uint32(),
			},
			Payload: randomPayload(4),
		},
	}

	messages := []*TestMessage{
		&TestMessage{
			TestHeader: chunks[0].TestHeader,
			Payload:    chunks[0].Payload,
		}, &TestMessage{
			TestHeader: chunks[1].TestHeader,
			Payload:    chunks[1].Payload,
		}, &TestMessage{
			TestHeader: chunks[2].TestHeader,
			Payload:    chunks[2].Payload,
		}, &TestMessage{
			TestHeader: chunks[3].TestHeader,
			Payload:    chunks[3].Payload,
		},
	}

	testChunkStream(chunks, messages, t)
}

func TestSameStream(t *testing.T) {
	chunks := []*TestChunk{
		&TestChunk{
			Fmt: HEADER_FMT_FULL,
			TestHeader: TestHeader{
				Timestamp:     rand.Uint32() & 0xffff,
				ChunkStreamID: 123,
				Length:        4,
				Type:          100,
				StreamID:      rand.Uint32(),
			},
			Payload: randomPayload(4),
		}, &TestChunk{
			Fmt: HEADER_FMT_FULL,
			TestHeader: TestHeader{
				Timestamp:     rand.Uint32() & 0xffff,
				ChunkStreamID: 124,
				Length:        4,
				Type:          101,
				StreamID:      rand.Uint32(),
			},
			Payload: randomPayload(4),
		}, &TestChunk{
			Fmt: HEADER_FMT_SAME_STREAM,
			TestHeader: TestHeader{
				Timestamp:     rand.Uint32() & 0xffff,
				ChunkStreamID: 123,
				Length:        4,
				Type:          102,
			},
			Payload: randomPayload(4),
		}, &TestChunk{
			Fmt: HEADER_FMT_SAME_STREAM,
			TestHeader: TestHeader{
				Timestamp:     rand.Uint32() & 0xffff,
				ChunkStreamID: 124,
				Length:        4,
				Type:          103,
			},
			Payload: randomPayload(4),
		},
	}

	messages := []*TestMessage{
		&TestMessage{
			TestHeader: chunks[0].TestHeader,
			Payload:    chunks[0].Payload,
		}, &TestMessage{
			TestHeader: chunks[1].TestHeader,
			Payload:    chunks[1].Payload,
		}, &TestMessage{
			TestHeader: TestHeader{
				Timestamp:     chunks[0].Timestamp + chunks[2].Timestamp,
				ChunkStreamID: chunks[2].ChunkStreamID,
				Length:        chunks[2].Length,
				Type:          chunks[2].Type,
				StreamID:      chunks[0].StreamID,
			},
			Payload: chunks[2].Payload,
		}, &TestMessage{
			TestHeader: TestHeader{
				Timestamp:     chunks[1].Timestamp + chunks[3].Timestamp,
				ChunkStreamID: chunks[3].ChunkStreamID,
				Length:        chunks[3].Length,
				Type:          chunks[3].Type,
				StreamID:      chunks[1].StreamID,
			},
			Payload: chunks[3].Payload,
		},
	}

	testChunkStream(chunks, messages, t)
}

func TestSameLengthAndStream(t *testing.T) {
	chunks := []*TestChunk{
		&TestChunk{
			Fmt: HEADER_FMT_FULL,
			TestHeader: TestHeader{
				Timestamp:     rand.Uint32() & 0xffff,
				ChunkStreamID: 123,
				Length:        4,
				Type:          10,
				StreamID:      rand.Uint32(),
			},
			Payload: randomPayload(4),
		}, &TestChunk{
			Fmt: HEADER_FMT_FULL,
			TestHeader: TestHeader{
				Timestamp:     rand.Uint32() & 0xffff,
				ChunkStreamID: 124,
				Length:        4,
				Type:          11,
				StreamID:      rand.Uint32(),
			},
			Payload: randomPayload(4),
		}, &TestChunk{
			Fmt: HEADER_FMT_SAME_LENGTH_AND_STREAM,
			TestHeader: TestHeader{
				Timestamp:     rand.Uint32() & 0xffff,
				ChunkStreamID: 123,
			},
			Payload: randomPayload(4),
		}, &TestChunk{
			Fmt: HEADER_FMT_SAME_LENGTH_AND_STREAM,
			TestHeader: TestHeader{
				Timestamp:     rand.Uint32() & 0xffff,
				ChunkStreamID: 124,
			},
			Payload: randomPayload(4),
		},
	}

	messages := []*TestMessage{
		&TestMessage{
			TestHeader: chunks[0].TestHeader,
			Payload:    chunks[0].Payload,
		}, &TestMessage{
			TestHeader: chunks[1].TestHeader,
			Payload:    chunks[1].Payload,
		}, &TestMessage{
			TestHeader: TestHeader{
				Timestamp:     chunks[0].Timestamp + chunks[2].Timestamp,
				ChunkStreamID: chunks[2].ChunkStreamID,
				Length:        chunks[0].Length,
				Type:          chunks[0].Type,
				StreamID:      chunks[0].StreamID,
			},
			Payload: chunks[2].Payload,
		}, &TestMessage{
			TestHeader: TestHeader{
				Timestamp:     chunks[1].Timestamp + chunks[3].Timestamp,
				ChunkStreamID: chunks[3].ChunkStreamID,
				Length:        chunks[1].Length,
				Type:          chunks[1].Type,
				StreamID:      chunks[1].StreamID,
			},
			Payload: chunks[3].Payload,
		},
	}

	testChunkStream(chunks, messages, t)
}

func TestContinuation(t *testing.T) {
	chunks := []*TestChunk{
		&TestChunk{
			Fmt: HEADER_FMT_FULL,
			TestHeader: TestHeader{
				Timestamp:     rand.Uint32() & 0xffff,
				ChunkStreamID: 123,
				Length:        4,
				Type:          10,
				StreamID:      rand.Uint32(),
			},
			Payload: randomPayload(4),
		}, &TestChunk{
			Fmt: HEADER_FMT_FULL,
			TestHeader: TestHeader{
				Timestamp:     rand.Uint32() & 0xffff,
				ChunkStreamID: 124,
				Length:        4,
				Type:          11,
				StreamID:      rand.Uint32(),
			},
			Payload: randomPayload(4),
		}, &TestChunk{
			Fmt: HEADER_FMT_CONTINUATION,
			TestHeader: TestHeader{
				Timestamp:     rand.Uint32() & 0xffff,
				ChunkStreamID: 123,
			},
			Payload: randomPayload(4),
		}, &TestChunk{
			Fmt: HEADER_FMT_CONTINUATION,
			TestHeader: TestHeader{
				Timestamp:     rand.Uint32() & 0xffff,
				ChunkStreamID: 124,
			},
			Payload: randomPayload(4),
		},
	}

	messages := []*TestMessage{
		&TestMessage{
			TestHeader: chunks[0].TestHeader,
			Payload:    chunks[0].Payload,
		}, &TestMessage{
			TestHeader: chunks[1].TestHeader,
			Payload:    chunks[1].Payload,
		}, &TestMessage{
			TestHeader: TestHeader{
				Timestamp:     chunks[0].Timestamp + chunks[0].Timestamp,
				ChunkStreamID: chunks[2].ChunkStreamID,
				Length:        chunks[0].Length,
				Type:          chunks[0].Type,
				StreamID:      chunks[0].StreamID,
			},
			Payload: chunks[2].Payload,
		}, &TestMessage{
			TestHeader: TestHeader{
				Timestamp:     chunks[1].Timestamp + chunks[1].Timestamp,
				ChunkStreamID: chunks[3].ChunkStreamID,
				Length:        chunks[1].Length,
				Type:          chunks[1].Type,
				StreamID:      chunks[1].StreamID,
			},
			Payload: chunks[3].Payload,
		},
	}

	testChunkStream(chunks, messages, t)
}

func TestLargeMessages(t *testing.T) {
	payload := randomPayload(int(3 * DEFAULT_CHUNK_SIZE))
	chunks := []*TestChunk{
		&TestChunk{
			Fmt: HEADER_FMT_FULL,
			TestHeader: TestHeader{
				Timestamp:     50,
				ChunkStreamID: 123,
				Length:        4,
				Type:          VIDEO_TYPE,
				StreamID:      1,
			},
			Payload: randomPayload(4),
		}, &TestChunk{
			Fmt: HEADER_FMT_FULL,
			TestHeader: TestHeader{
				Timestamp:     100,
				ChunkStreamID: 123,
				Length:        3 * DEFAULT_CHUNK_SIZE,
				Type:          VIDEO_TYPE,
				StreamID:      1,
			},
			Payload: payload[:DEFAULT_CHUNK_SIZE],
		}, &TestChunk{
			Fmt: HEADER_FMT_CONTINUATION,
			TestHeader: TestHeader{
				Timestamp:     100,
				ChunkStreamID: 123,
			},
			Payload: payload[DEFAULT_CHUNK_SIZE : 2*DEFAULT_CHUNK_SIZE],
		}, &TestChunk{
			Fmt: HEADER_FMT_CONTINUATION,
			TestHeader: TestHeader{
				Timestamp:     100,
				ChunkStreamID: 123,
			},
			Payload: payload[2*DEFAULT_CHUNK_SIZE:],
		}, &TestChunk{
			Fmt: HEADER_FMT_SAME_STREAM,
			TestHeader: TestHeader{
				Timestamp:     101,
				ChunkStreamID: 123,
				Length:        4,
				Type:          VIDEO_TYPE,
			},
			Payload: randomPayload(4),
		},
	}

	messages := []*TestMessage{
		&TestMessage{
			TestHeader: chunks[0].TestHeader,
			Payload:    chunks[0].Payload,
		}, &TestMessage{
			TestHeader: chunks[1].TestHeader,
			Payload:    payload,
		}, &TestMessage{
			TestHeader: TestHeader{
				ChunkStreamID: chunks[4].ChunkStreamID,
				Timestamp:     chunks[1].Timestamp + chunks[4].Timestamp, // chunk[1] is the most recent absolute, [4] is a delta
				Length:        chunks[4].Length,
				Type:          chunks[4].Type,
				StreamID:      chunks[1].StreamID,
			},
			Payload: chunks[4].Payload,
		},
	}

	testChunkStream(chunks, messages, t)
}

func TestSetChunkSize(t *testing.T) {
	chunkSize := make([]byte, 4)
	binary.BigEndian.PutUint32(chunkSize, DEFAULT_CHUNK_SIZE*2)

	chunks := []*TestChunk{
		&TestChunk{
			Fmt: HEADER_FMT_FULL,
			TestHeader: TestHeader{
				Timestamp:     rand.Uint32() & 0xffff,
				ChunkStreamID: 123,
				Length:        DEFAULT_CHUNK_SIZE * 2,
				Type:          100,
				StreamID:      rand.Uint32(),
			},
			Payload: randomPayload(int(DEFAULT_CHUNK_SIZE)),
		}, &TestChunk{
			Fmt: HEADER_FMT_CONTINUATION,
			TestHeader: TestHeader{
				Timestamp:     rand.Uint32() & 0xffff,
				ChunkStreamID: 123,
			},
			Payload: randomPayload(int(DEFAULT_CHUNK_SIZE)),
		}, &TestChunk{
			Fmt: HEADER_FMT_FULL,
			TestHeader: TestHeader{
				ChunkStreamID: CS_ID_PROTOCOL_CONTROL,
				Length:        4,
				Type:          SET_CHUNK_SIZE,
			},
			Payload: chunkSize,
		}, &TestChunk{
			Fmt: HEADER_FMT_FULL,
			TestHeader: TestHeader{
				Timestamp:     rand.Uint32() & 0xffff,
				ChunkStreamID: 123,
				Length:        DEFAULT_CHUNK_SIZE * 2,
				Type:          101,
				StreamID:      rand.Uint32(),
			},
			Payload: randomPayload(int(DEFAULT_CHUNK_SIZE * 2)),
		},
	}

	messages := []*TestMessage{
		&TestMessage{
			TestHeader: chunks[0].TestHeader,
			Payload:    append(chunks[0].Payload, chunks[1].Payload...),
		}, &TestMessage{
			TestHeader: chunks[2].TestHeader,
			Payload:    chunks[2].Payload,
		}, &TestMessage{
			TestHeader: chunks[3].TestHeader,
			Payload:    chunks[3].Payload,
		},
	}

	testChunkStream(chunks, messages, t)
}

func TestInvalidStreamStart(t *testing.T) {
	errorChunks := []*TestChunk{
		&TestChunk{
			Fmt: HEADER_FMT_SAME_STREAM,
			TestHeader: TestHeader{
				ChunkStreamID: randomChunkStreamID(),
				Timestamp:     randomTimestamp(),
				Type:          randomType(),
				Length:        4,
				StreamID:      randomStreamID(),
			},
			Payload: randomPayload(4),
		}, &TestChunk{
			Fmt: HEADER_FMT_SAME_LENGTH_AND_STREAM,
			TestHeader: TestHeader{
				ChunkStreamID: randomChunkStreamID(),
				Timestamp:     randomTimestamp(),
				Type:          randomType(),
				Length:        4,
				StreamID:      randomStreamID(),
			},
			Payload: randomPayload(4),
		}, &TestChunk{
			Fmt: HEADER_FMT_CONTINUATION,
			TestHeader: TestHeader{
				ChunkStreamID: randomChunkStreamID(),
				Timestamp:     randomTimestamp(),
				Type:          randomType(),
				Length:        4,
				StreamID:      randomStreamID(),
			},
			Payload: randomPayload(4),
		},
	}

	for _, chunk := range errorChunks {
		buf := &bytes.Buffer{}
		writeChunks(buf, []*TestChunk{chunk})
		csr := NewChunkStreamReader(buf)
		msg, err := csr.Read()
		if err == nil {
			t.Fatalf("Chunk %#v should trigger a read error, got %#v instead", chunk, msg)
		}
	}
}

func TestExtendedTimestamp(t *testing.T) {
	chunks := []*TestChunk{
		&TestChunk{
			Fmt: HEADER_FMT_FULL,
			TestHeader: TestHeader{
				ChunkStreamID: 100,
				Timestamp:     randomExtendedTimestamp(),
				Type:          randomType(),
				Length:        4,
				StreamID:      1,
			},
			Payload: randomPayload(4),
		}, &TestChunk{
			Fmt: HEADER_FMT_SAME_STREAM,
			TestHeader: TestHeader{
				ChunkStreamID: 100,
				Timestamp:     1,
				Type:          randomType(),
				Length:        4,
			},
			Payload: randomPayload(4),
		},
	}

	messages := []*TestMessage{
		&TestMessage{
			TestHeader: chunks[0].TestHeader,
			Payload:    chunks[0].Payload,
		}, &TestMessage{
			TestHeader: TestHeader{
				ChunkStreamID: chunks[0].ChunkStreamID,
				Timestamp:     chunks[0].Timestamp + chunks[1].Timestamp,
				Type:          chunks[1].Type,
				Length:        chunks[1].Length,
				StreamID:      chunks[0].StreamID,
			},
			Payload: chunks[1].Payload,
		},
	}

	testChunkStream(chunks, messages, t)
}

func TestContinuationExtendedTimestamp(t *testing.T) {
	extTs := uint32(0xFFFFFF + 1) // minimum extended timestamp
	payload := randomPayload(int(DEFAULT_CHUNK_SIZE * 3))

	chunks := []*TestChunk{
		// This first set tests continuation chunks that are part
		// of an incomplete message
		&TestChunk{
			Fmt: HEADER_FMT_FULL,
			TestHeader: TestHeader{
				ChunkStreamID: 100,
				Timestamp:     extTs,
				Type:          randomType(),
				Length:        DEFAULT_CHUNK_SIZE * 3,
				StreamID:      1,
			},
			Payload: payload[0:DEFAULT_CHUNK_SIZE],
		},
		&TestChunk{
			Fmt: HEADER_FMT_CONTINUATION,
			TestHeader: TestHeader{
				ChunkStreamID: 100,
				Timestamp:     extTs + 1, // this timestamp will be written but ingored
			},
			Payload: payload[DEFAULT_CHUNK_SIZE : DEFAULT_CHUNK_SIZE*2],
		},
		&TestChunk{
			Fmt: HEADER_FMT_CONTINUATION,
			TestHeader: TestHeader{
				ChunkStreamID: 100,
				Timestamp:     extTs + 2, // this timestamp will be written but ignored
			},
			Payload: payload[DEFAULT_CHUNK_SIZE*2:],
		},
		&TestChunk{
			Fmt: HEADER_FMT_CONTINUATION,
			TestHeader: TestHeader{
				ChunkStreamID: 100,
				Timestamp:     extTs + 1, // this timestamp will be used because it is ext
			},
			Payload: payload[0:DEFAULT_CHUNK_SIZE],
		},
		&TestChunk{
			Fmt: HEADER_FMT_CONTINUATION,
			TestHeader: TestHeader{
				ChunkStreamID: 100,
				Timestamp:     extTs + 2, // this timestamp will be written but ingored
			},
			Payload: payload[DEFAULT_CHUNK_SIZE : DEFAULT_CHUNK_SIZE*2],
		},
		&TestChunk{
			Fmt: HEADER_FMT_CONTINUATION,
			TestHeader: TestHeader{
				ChunkStreamID: 100,
				Timestamp:     extTs + 3, // this timestamp will be written but ignored
			},
			Payload: payload[DEFAULT_CHUNK_SIZE*2:],
		},
	}

	messages := []*TestMessage{
		&TestMessage{
			TestHeader: chunks[0].TestHeader,
			Payload:    payload,
		},
		&TestMessage{
			TestHeader: TestHeader{
				ChunkStreamID: chunks[0].ChunkStreamID,
				Timestamp:     extTs + extTs,
				Type:          chunks[0].Type,
				Length:        chunks[0].Length,
				StreamID:      chunks[0].StreamID,
			},
			Payload: payload,
		},
	}

	if testing.Verbose() {
		t.Logf("Chunks:")
		for _, c := range chunks {
			t.Logf("%#v", c)
		}
		t.Logf("Messages:")
		for _, m := range messages {
			t.Logf("%#v", m)
		}
	}

	testChunkStream(chunks, messages, t)
}
