package rtmp

import (
	"testing"
)

var (
	flvH264SequenceHeader = []byte{
		0x17,             // Frame Type, Codec
		0x00,             // AVCPacketType
		0x00, 0x00, 0x00, // CTS
		// Insert extradata here
	}
	flvH264Nalu = []byte{
		0x17,             // Frame Type, Codec
		0x01,             // AVCPacketType
		0x00, 0x00, 0x00, // CTS
		// Insert NALU(s) here
	}
	flvVP6Frame = []byte{
		0x14, // Frame Type, Codec
		// Insert VP6 data here
	}
	flvAACSequenceHeader = []byte{
		0xAF, // Format, Rate, Size, Type
		0x00, // AACPacketType
	}
	flvAACFrame = []byte{
		0xAF, // Format, Rate, Size, Type
		0x01, // AACPacketType
	}
	flvMP3Frame = []byte{
		0x2F, // Format, Rate, Size, Type
	}
	flvSpeexFrame = []byte{
		0xB2, // Format, Rate, Size, Type
	}
)

func newTag(tagType uint8, timestamp uint32, payload []byte) *FlvTag {
	return &FlvTag{
		Type:      tagType,
		Timestamp: timestamp,
		Size:      uint32(len(payload)),
		Bytes:     payload,
	}
}

func checkVideoIsNotAudio(t *testing.T, tag *FlvTag) {
	ah, err := tag.GetAudioHeader()
	if ah != nil || err != ErrPacketTypeMismatch {
		t.Fatalf("Parsed video packet as audio %v %v", ah, err)
	}
}

func checkAudioIsNotVideo(t *testing.T, tag *FlvTag) {
	ah, err := tag.GetVideoHeader()
	if ah != nil || err != ErrPacketTypeMismatch {
		t.Fatalf("Parsed audio packet as video %v %v", ah, err)
	}
}

func TestH264SequenceHeader(t *testing.T) {
	tag := newTag(VIDEO_TYPE, 0, flvH264SequenceHeader)
	vh, err := tag.GetVideoHeader()
	if vh == nil || err != nil {
		t.Fatalf("Failed to parse H.264 video header %v %v", vh, err)
	}
	checkVideoIsNotAudio(t, tag)
}

func TestH264Nalu(t *testing.T) {
	tag := newTag(VIDEO_TYPE, 0, flvH264Nalu)
	vh, err := tag.GetVideoHeader()
	if vh == nil || err != nil {
		t.Fatalf("Failed to parse H.264 video frame %v %v", vh, err)
	}
	checkVideoIsNotAudio(t, tag)
}

func TestTruncatedH264(t *testing.T) {
	tag := newTag(VIDEO_TYPE, 0, flvH264Nalu[:1])
	vh, err := tag.GetVideoHeader()
	if vh != nil || err != ErrTruncatedPacket {
		t.Fatalf("Parsed an H.264 packet without an AVCPacketType %v %v", vh, err)
	}
	checkVideoIsNotAudio(t, tag)
}

func TestVp6Video(t *testing.T) {
	tag := newTag(VIDEO_TYPE, 0, flvVP6Frame)
	vh, err := tag.GetVideoHeader()
	if vh == nil || err != nil {
		t.Fatalf("Failed to parse non-H264 video tag %v %v", vh, err)
	}
	checkVideoIsNotAudio(t, tag)
}

func TestEmptyVideo(t *testing.T) {
	tag := newTag(VIDEO_TYPE, 0, []byte{})
	vh, err := tag.GetVideoHeader()
	if vh != nil || err != ErrTruncatedPacket {
		t.Fatalf("Parsed a video header out of empty data %v %v", vh, err)
	}
	checkVideoIsNotAudio(t, tag)
}

func TestAACSequenceHeader(t *testing.T) {
	tag := newTag(AUDIO_TYPE, 0, flvAACSequenceHeader)
	ah, err := tag.GetAudioHeader()
	if ah == nil || err != nil {
		t.Fatalf("Failed to parse AAC sequence header %v %v", ah, err)
	}
	checkAudioIsNotVideo(t, tag)
}

func TestAACFrame(t *testing.T) {
	tag := newTag(AUDIO_TYPE, 0, flvAACFrame)
	ah, err := tag.GetAudioHeader()
	if ah == nil || err != nil {
		t.Fatalf("Failed to parse AAC frame %v %v", ah, err)
	}
	checkAudioIsNotVideo(t, tag)
}

func TestTruncatedAACFrame(t *testing.T) {
	tag := newTag(AUDIO_TYPE, 0, flvAACFrame[:1])
	ah, err := tag.GetAudioHeader()
	if ah != nil || err != ErrTruncatedPacket {
		t.Fatalf("Parsed an AAC packet without an AACPacketType %v %v", ah, err)
	}
	checkAudioIsNotVideo(t, tag)
}

func TestMP3Frame(t *testing.T) {
	tag := newTag(AUDIO_TYPE, 0, flvMP3Frame)
	ah, err := tag.GetAudioHeader()
	if ah == nil || err != nil {
		t.Fatalf("Failed to parse MP3 frame %v %v", ah, err)
	}
	checkAudioIsNotVideo(t, tag)
}

func TestSpeexFrame(t *testing.T) {
	tag := newTag(AUDIO_TYPE, 0, flvSpeexFrame)
	ah, err := tag.GetAudioHeader()
	if ah == nil || err != nil {
		t.Fatalf("Failed to parse Speex frame %v %v", ah, err)
	}
	checkAudioIsNotVideo(t, tag)
}

func TestEmptyAudio(t *testing.T) {
	tag := newTag(AUDIO_TYPE, 0, []byte{})
	vh, err := tag.GetAudioHeader()
	if vh != nil || err != ErrTruncatedPacket {
		t.Fatalf("Parsed an audio header out of empty data %v %v", vh, err)
	}
	checkAudioIsNotVideo(t, tag)
}

func TestCommandMessage(t *testing.T) {
	tag := newTag(COMMAND_AMF0, 0, []byte{0x00, 0x00, 0x00})
	ah, err := tag.GetAudioHeader()
	if ah != nil || err != ErrPacketTypeMismatch {
		t.Fatal("Parsed command message as audio")
	}
	vh, err := tag.GetVideoHeader()
	if vh != nil || err != ErrPacketTypeMismatch {
		t.Fatal("Parsed command message as video")
	}
}

func TestZeroValueMessage(t *testing.T) {
	tag := newTag(0, 0, []byte{})
	ah, err := tag.GetAudioHeader()
	if ah != nil || err != ErrPacketTypeMismatch {
		t.Fatal("Parsed zero message as audio")
	}
	vh, err := tag.GetVideoHeader()
	if vh != nil || err != ErrPacketTypeMismatch {
		t.Fatal("Parsed zero message as video")
	}
}
