package rtmp

import (
	"bytes"
	"code.justin.tv/video/gortmp/pkg/log"
	"context"
	"sync"
)

type MediaPlayer interface {
	Stream() Stream
	Start()
	Close()
	Wait()
}

type mediaPlayer struct {
	mb MediaBuffer
	ms MediaStream
	s  Stream

	done chan bool

	loopOnce sync.Once

	log log.Logger
}

func NewMediaPlayer(ctx context.Context, mediaStream MediaStream, stream Stream) (MediaPlayer, error) {
	// default to a 4mb buffer
	mb, err := NewMediaBuffer(ctx, mediaStream, 4*1024*1024)
	if err != nil {
		return nil, err
	}

	mp := &mediaPlayer{
		mb:   mb,
		ms:   mediaStream,
		s:    stream,
		done: make(chan bool),
		log:  log.FromContext(ctx, "mediaplayer"),
	}

	return mp, nil
}

func (mp *mediaPlayer) Stream() Stream {
	return mp.s
}

func (mp *mediaPlayer) loop() {
	mp.log.Debugf("start")
	defer mp.log.Debugf("stop")
	defer mp.Close()
	defer func() {
		close(mp.done)
	}()

	for {
		tag, err := mp.mb.Get()
		if err != nil {
			if err != MediaStreamClosed {
				mp.log.Errorf("error getting next tag: %s", err)
			}
			return
		}

		err = mp.writeTag(tag)
		if err != nil {
			mp.log.Errorf("error writing tag: %s", err)
			return
		}
	}
}

func (mp *mediaPlayer) writeTag(tag *FlvTag) error {
	return mp.s.Write(&RawMessage{
		Type:      tag.Type,
		Timestamp: tag.Timestamp,
		Data:      bytes.NewBuffer(tag.Bytes),
	})
}

func (mp *mediaPlayer) Start() {
	mp.loopOnce.Do(func() {
		go mp.loop()
	})
}

func (mp *mediaPlayer) Close() {
	mp.mb.Close()
	mp.s.Conn().Close()
}

func (mp *mediaPlayer) Wait() {
	<-mp.done
}
