// +build ignore

package main

import (
	"context"
	"crypto/tls"
	"crypto/x509"
	"flag"
	"io/ioutil"
	"log"
	"net/http"

	"code.justin.tv/video/lvsapi/rpc/lvs"
)

var (
	serverCert              = flag.String("server-cert", "", "server public certificate")
	clientCert              = flag.String("client-cert", "", "client certificate")
	clientKey               = flag.String("client-key", "", "client key")
	serverUrl               = flag.String("server-url", "", "base url for the lvsapi twirp server")
	contentId               = flag.String("content-id", "", "content id")
	s3Prefix                = flag.String("s3-prefix", "", "s3 prefix")
	s3Bucket                = flag.String("s3-bucket", "", "s3 bucket")
	snsNotificationEndpoint = flag.String("sns-notification-endpoint", "", "sns")
)

func main() {
	log.SetFlags(0)
	flag.Parse()

	// Load client cert
	cert, err := tls.LoadX509KeyPair(*clientCert, *clientKey)
	if err != nil {
		log.Fatal("Failed to load client key pair: ", err)
	}

	var certPool *x509.CertPool

	if *serverCert != "" {
		// Load CA cert
		caCert, err := ioutil.ReadFile(*serverCert)
		if err != nil {
			log.Fatal("Failed to read server cert: ", err)
		}
		certPool = x509.NewCertPool()
		certPool.AppendCertsFromPEM(caCert)
	}

	// Setup HTTPS client
	tlsConfig := &tls.Config{
		Certificates: []tls.Certificate{cert},
		RootCAs:      certPool,
	}
	tlsConfig.BuildNameToCertificate()
	transport := &http.Transport{TLSClientConfig: tlsConfig}
	client := &http.Client{Transport: transport}

	lvsClient := lvs.NewLiveVideoServiceProtobufClient(*serverUrl, client)
	resp, err := lvsClient.CreateStreamKey(context.Background(), &lvs.CreateStreamKeyRequest{
		ContentId:               *contentId,
		S3Prefix:                *s3Prefix,
		S3Bucket:                *s3Bucket,
		SnsNotificationEndpoint: *snsNotificationEndpoint,
	})
	if err != nil {
		log.Fatal("Twirp error: ", err)
	}
	log.Printf("RTMP URL: %s", resp.GetRtmpIngestUrl())
	log.Printf("PLAYBACK URL: %s", resp.GetPlaybackUrl())
}
