package main

import (
	"bytes"
	"flag"
	"fmt"
	"io/ioutil"
	"log"
	"net"
	"net/http"
	"os"
	"strings"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/ec2metadata"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/pkg/errors"
	"github.com/twitchtv/twirp/hooks/statsd"

	goji "goji.io"
	"goji.io/pat"

	"code.justin.tv/common/gometrics"
	"code.justin.tv/video/lvsapi/internal/logging"
	"code.justin.tv/video/lvsapi/internal/metrics"
	"code.justin.tv/video/lvsapi/internal/server"
	"code.justin.tv/video/lvsapi/internal/utils"

	"code.justin.tv/video/lvsapi/streamkey"
)

var (
	portInternal     = flag.String("port", "8080", "api service port")
	secretS3Bucket   = flag.String("secret-s3-bucket", "", "s3 bucket to use for streamkey secret storage")
	secretS3Prefix   = flag.String("secret-s3-prefix", "", "s3 bucket prefix to use when determining streamkey secret key names")
	secretS3Duration = flag.Duration("secret-s3-duration", 60*time.Second, "specifies how long to cache secrets in between requests to s3")
	twitchEnv        = flag.String("twitch-env", "production", "production/staging/dev")
	statsdServer     = flag.String("statsd-server", "graphite.internal.justin.tv:8125", "The address of the statsd server where we want to send stats")
	disableMWS       = flag.Bool("disable-mws", false, "when set to true metrics will not be sent to MWS")
)

var envMap = map[string]string{
	"port":               "LVSAPI_PORT",
	"secret-s3-bucket":   "LVSAPI_SECRET_S3_BUCKET",
	"secret-s3-prefix":   "LVSAPI_SECRET_S3_PREFIX",
	"secret-s3-duration": "LVSAPI_SECRET_S3_DURATION",
	"twitch-env":         "TWITCH_ENV",
	"statsd-server":      "LVSAPI_STATSD_SERVER",
	"disable-mws":        "LVSAPI_DISABLE_MWS",
}

func mustNotErr(err error) {
	if err != nil {
		panic(err)
	}
}

func awsSession() (*session.Session, error) {
	metaSession, err := session.NewSession()
	if err != nil {
		return nil, errors.Wrap(err, "failed to create aws session")
	}
	metaClient := ec2metadata.New(metaSession)

	// we explicitly ignore this error because if this call errors we're just
	// not running on an EC2 instance
	region, _ := metaClient.Region()

	sess, err := session.NewSession(&aws.Config{Region: aws.String(region)})
	if err != nil {
		return nil, errors.Wrap(err, "failed to create aws session")
	}

	return sess, nil
}

func configureServer(h http.Handler) *http.Server {
	return &http.Server{
		Addr:    net.JoinHostPort("", *portInternal),
		Handler: h,
	}
}

func configureHandler(logger logging.Logger, s statsd.Statter) http.Handler {
	mux := goji.NewMux()

	mux.Use(logging.Middleware(logger))

	// simple "is this server on" healthcheck
	mux.Handle(pat.Get("/debug/health"), http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		_, _ = w.Write([]byte("We Gucci"))
	}))

	sess, sessErr := awsSession()
	var secretSource streamkey.SecretSource

	if *secretS3Bucket == "" {
		if strings.ToLower(*twitchEnv) == "production" || strings.ToLower(*twitchEnv) == "staging" {
			logger.Fatal("S3 Bucket for streamkey secrets missing")
		}

		// For dev environment, use single secret source
		logger.Debug("Autogenerating secret as no s3 bucket was specified")
		secret, err := streamkey.GenerateSecret()
		if err != nil {
			logger.WithError(err).Fatal("failed to generate secret")
		}

		secretSource = &streamkey.SingleSecretSource{Secret: secret}
	} else {
		if sessErr != nil {
			logger.WithError(sessErr).Fatal("failed to create aws session")
		}

		secretSource = streamkey.NewS3SecretSource(streamkey.S3SecretSourceConfig{
			S3:           s3.New(sess),
			Bucket:       *secretS3Bucket,
			Prefix:       *secretS3Prefix,
			CacheTimeout: *secretS3Duration,
		})
	}

	// Read the credentials from the env or from the session if there is one
	creds := credentials.NewSharedCredentials("", "")
	validSession := sess != nil && sess.Config != nil && sess.Config.Credentials != nil
	if validSession {
		creds = sess.Config.Credentials
	}
	m := metrics.New("lvsapi", "", *twitchEnv, creds)
	if !*disableMWS && validSession {
		m.Start()
	}

	// all other paths go to pkg server's handler
	mux.Handle(pat.New("/*"), server.NewInternal(secretSource, s, m))

	return mux
}

func runServer(s *http.Server) chan error {
	ch := make(chan error, 1)
	go func() {
		defer close(ch)
		ch <- s.ListenAndServe()
	}()
	return ch
}

func reqLogger(base http.Handler) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		body, err := ioutil.ReadAll(r.Body)
		if err != nil {
			// Broken HTTP connection
			w.WriteHeader(500)
			return
		}
		log.Printf("request headers: %+v", r.Header)
		log.Printf("request body: %s", body)

		r.Body = ioutil.NopCloser(bytes.NewBuffer(body))
		base.ServeHTTP(w, r)
	})
}

func mustParseFlags() {
	flag.Parse()
	missing, err := utils.ValidateFlags(envMap)
	mustNotErr(err)

	*twitchEnv = strings.ToLower(*twitchEnv)
	if len(missing) != 0 && (*twitchEnv == "production" || *twitchEnv == "staging") {
		panic(fmt.Errorf("Missing flags: %v", missing))
	}
}

func main() {
	mustParseFlags()

	logger := logging.New(logging.Config{})
	logger.Info("LiveVideoServiceInternal API")

	// Create the statsd client and pass it around to the configureHandler
	s, err := metrics.NewStatter(fmt.Sprintf("lvsapi.%s", *twitchEnv), *statsdServer)
	if err != nil {
		logger.WithError(err).Errorf("Failed to connect to statsd: %v", err)
	}

	// Collect Golang metrics periodically
	gometrics.Monitor(s, 10*time.Second)

	// handler to be used by both http and https listener
	handler := reqLogger(configureHandler(logger, s))

	// set up our servers
	httpServer := configureServer(handler)
	httpErrCh := runServer(httpServer)

	select {
	case err := <-httpErrCh:
		logger.WithError(err).Error("http listener failed")
		os.Exit(1)
	}
}
