package main

import (
	"context"
	"flag"
	"fmt"
	"io"
	"io/ioutil"
	"net"
	"net/http"
	"testing"

	"code.justin.tv/video/lvsapi/internal/logging"
	"code.justin.tv/video/lvsapi/rpc/lvs"
	"github.com/stretchr/testify/require"
	"github.com/twitchtv/twirp"
)

func TestHttpServer(t *testing.T) {
	require.NoError(t, flag.Set("twitch-env", "dev"))
	handler := configureHandler(logging.Silent(), nil)
	srv := configureServer(handler)
	ln, err := net.Listen("tcp", ":0")
	require.NoError(t, err)

	done := make(chan error, 1)
	go func() {
		done <- srv.Serve(ln)
	}()

	port := ln.Addr().(*net.TCPAddr).Port
	// query health endpoint
	resp, err := http.Get(fmt.Sprintf("http://localhost:%d/debug/health", port))
	require.NoError(t, err)
	_, err = io.Copy(ioutil.Discard, resp.Body)
	require.NoError(t, err)
	err = resp.Body.Close()
	require.NoError(t, err)

	// make an authorize broadcast query against the lvs api to make sure it's installed
	lvsClient := lvs.NewLiveVideoServiceInternalProtobufClient(fmt.Sprintf("http://localhost:%d", port), http.DefaultClient)

	// this is expected to fail to authorize streamkey. detailed tests are at the service level.
	// here we will just make sure that expected error is thrown
	_, err = lvsClient.AuthorizeBroadcast(context.Background(), &lvs.AuthorizeBroadcastRequest{})
	require.Error(t, err)
	require.EqualValues(t, err, twirp.NewError(twirp.Unauthenticated, "failed to authorize streamkey"))

	_ = srv.Close()
	<-done
}
