import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.security.KeyPair;
import java.security.KeyStore;
import java.security.PrivateKey;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;

import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManagerFactory;

import org.apache.commons.io.IOUtils;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.bouncycastle.openssl.PEMReader;
import org.json.JSONObject;

public class LvsClient {
	private String serverCert, clientCert, clientKey;

	public LvsClient(String serverCert, String clientCert, String clientKey) {
		this.serverCert = serverCert;
		this.clientCert = clientCert;
		this.clientKey = clientKey;
	}

	private PrivateKey parseClientKey() throws IOException {
		FileReader fr = null;
		BufferedReader br = null;
		PEMReader reader = null;

		try {
			fr = new FileReader(clientKey);
			br = new BufferedReader(fr);
			reader = new PEMReader(br);
			Object o = reader.readObject();
			if (!(o instanceof KeyPair)) {
				throw new IOException("LvsClient.parseClientKey(): no keypair found, got type "+o.getClass());
			}
			KeyPair pair = (KeyPair)o;
			return pair.getPrivate();
		} finally {
			if (fr != null) fr.close();
			if (br != null) br.close();
			if (reader != null) reader.close();
		}
	}

	private Certificate parseClientCert() throws IOException, CertificateException {
		return parseCert(clientCert);
	}

	private Certificate parseServerCert() throws IOException, CertificateException {
		return parseCert(serverCert);
	}

	private Certificate parseCert(String path) throws IOException, CertificateException {
		FileInputStream fis = null;
		BufferedInputStream bis = null;

		try {
			fis = new FileInputStream(path);
			bis = new BufferedInputStream(fis);

			CertificateFactory cf = CertificateFactory.getInstance("X.509");
			Certificate cert = cf.generateCertificate(bis);
			return cert;
		} finally {
			if (fis != null) fis.close();
			if (bis != null) bis.close();
		}
	}

	private CloseableHttpClient buildHttpClient() throws Exception {
		PrivateKey clientKey = parseClientKey();
		Certificate clientCert = parseClientCert();
		Certificate serverCert = null;
		if (this.serverCert != null) {
			serverCert = parseServerCert();
		}

		Certificate[] clientChain = {clientCert};

		// this sets the client private keys up
		KeyStore clientKeyStore = KeyStore.getInstance("JKS");
		clientKeyStore.load(null, new char[0]);
		clientKeyStore.setKeyEntry("service-tls", clientKey, new char[0], clientChain);

		// this sets the server public key so the client can ensure they are speaking with a valid server
		KeyStore clientTrustStore = KeyStore.getInstance("JKS");
		clientTrustStore.load(null, new char[0]);
		clientTrustStore.setCertificateEntry("server-tls", serverCert);

		KeyManagerFactory kmf = KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm());
		kmf.init(clientKeyStore, new char[0]);

		TrustManagerFactory tmf = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
		tmf.init(clientTrustStore);

		SSLContext sslContext = SSLContext.getInstance("TLS");
		sslContext.init(kmf.getKeyManagers(), tmf.getTrustManagers(), null);
		CloseableHttpClient client = HttpClients.custom().setSSLContext(sslContext).build();
		return client;
	}

	private HttpPost buildCheckAuthRequest() {
		HttpPost post = new HttpPost("https://lvs.twitch.tv/twirp/code.justin.tv.video.lvsapi.rpc.lvs.LiveVideoService/CheckAuth");
		post.addHeader("Content-Type", "application/json");
		post.setEntity(new ByteArrayEntity("{}".getBytes()));
		return post;
	}

	private JSONObject readCheckAuthResponse(CloseableHttpResponse response) throws IOException {
		InputStream content = response.getEntity().getContent();
		try {
			return new JSONObject(IOUtils.toString(content));
		} finally {
			content.close();
		}
	}

	public boolean checkAuth() throws Exception {
		CloseableHttpClient client = this.buildHttpClient();
		CloseableHttpResponse response = null;

		try {
			response = client.execute(this.buildCheckAuthRequest());
		} catch (Exception e) {
			System.out.println("HTTP Client error: "+e.getMessage());
			return false;
		}

		JSONObject obj = null;

		try {
			obj = this.readCheckAuthResponse(response);
			System.out.println(""+obj);
		} finally {
			response.close();
		}

		if (obj == null) {
			System.out.println("Received empty service response");
			return false;
		}

		return true;
	}
}
