package auth

import (
	"context"
	"net/http"
	"regexp"
)

const (
	// ForceAuthHeader is used to force authentication of a specific customer. It is not
	// for use in production, certificate auth is the appropriate mechanism for that
	ForceAuthHeader     = "X-Twitch-Lvs-Force-Auth"
	maxCustomerIDLength = 40
)

type lvsCustomerIDKey struct{}

// contentIDRegex defines the format of a content Identifier
var customerIDRegex = regexp.MustCompile(`([A-Za-z0-9\-\_]+)`)

// IsValidCustomerID verifies that the passed in CustomerID conforms to our customer id specification
func IsValidCustomerID(customerID string) bool {
	parsed := customerIDRegex.FindAllString(customerID, -1)

	if len(parsed) != 1 {
		return false
	}

	// ContentID Regex should only find a single match.
	if parsed[0] == customerID && (len(parsed[0]) <= maxCustomerIDLength) {
		return true
	}

	return false
}

// FromCertificate is a middleware function that may be used directly or with a tool like goji to extract
// authentication information from a client certificate (or other means for dev purposes)
func FromCertificate(h http.Handler) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		h.ServeHTTP(w, requestAuthFromCertificate(r))
	})
}

// FromHeader is a middleware function that may be used directly or with a tool like goji to extract
// authentication information from request headers. This is intended for use in development
func FromHeader(h http.Handler) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		h.ServeHTTP(w, requestAuthFromHeader(r))
	})
}

// RequireValid is a midldlware that will reject all requests with invalid credentials
func RequireValid(h http.Handler) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		customerID, ok := LvsCustomerID(r.Context())
		if !ok || !IsValidCustomerID(customerID) {
			http.NotFound(w, r)
		} else {
			h.ServeHTTP(w, r)
		}

	})
}

// LvsCustomerID returns the customer ID which was retrieved from the client's certificate if any
func LvsCustomerID(ctx context.Context) (string, bool) {
	val, ok := ctx.Value(lvsCustomerIDKey{}).(string)
	return val, ok
}

func setLvsCustomerID(ctx context.Context, id string) context.Context {
	return context.WithValue(ctx, lvsCustomerIDKey{}, id)
}

func requestAuthFromCertificate(r *http.Request) *http.Request {
	if r.TLS == nil {
		return r
	}

	if len(r.TLS.PeerCertificates) != 1 {
		return r
	}

	customerID := r.TLS.PeerCertificates[0].Subject.CommonName
	ctx := setLvsCustomerID(r.Context(), customerID)
	return r.WithContext(ctx)
}

func requestAuthFromHeader(r *http.Request) *http.Request {
	customerID := r.Header.Get(ForceAuthHeader)
	ctx := setLvsCustomerID(r.Context(), customerID)
	return r.WithContext(ctx)
}

// AddSecurityHeader adds security headers to the responses from our service
func AddSecurityHeader(h http.Handler) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		// Content Security Policy (CSP) is an added layer of security that helps
		// to detect and mitigate certain types of attacks, including
		// Cross Site Scripting (XSS) and data injection attacks.
		// Since we don't host the website in this case and dont return any JS,
		// we should block all attempts to do so

		w.Header().Set("Content-Security-Policy", "default-src: 'none'")
		w.Header().Set("X-Frame-Options", "DENY")

		h.ServeHTTP(w, r)
	})
}
