package caching

import (
	"context"
	"fmt"
	"strings"
	"time"

	"github.com/cactus/go-statsd-client/statsd"
	"github.com/patrickmn/go-cache"

	"code.justin.tv/video/lvsapi/internal/caching/gocache"
	"code.justin.tv/video/lvsapi/internal/logging"
	"code.justin.tv/video/lvsapi/internal/usher"
	"github.com/twitchtv/twirp"
)

// InMemoryCache wraps the cache implementation and usher backend
type InMemoryCache struct {
	usher             usher.UsherBackend
	cacheImpl         gocache.GoCache
	defaultExpiration time.Duration
	statter           statsd.Statter
}

// The key for cache will be the customer_id string
type cacheValue map[string]usher.UsherStreamResponse

// NewInMemoryCache returns a Cache object
func NewInMemoryCache(expiration, purgeInterval *time.Duration, usherBackend usher.UsherBackend, s statsd.Statter) CacheInterface {
	return &InMemoryCache{
		usher:             usherBackend,
		cacheImpl:         cache.New(*expiration, *purgeInterval),
		defaultExpiration: *expiration,
		statter:           s,
	}
}

// GetStream returns stream entry for customer id and content id
func (c *InMemoryCache) GetStream(ctx context.Context, customerId, contentId string) (*usher.UsherStreamResponse, error) {
	customerStreams, err := c.get(ctx, customerId)
	if err != nil {
		return nil, err
	}

	if contentStream, ok := customerStreams[contentId]; ok {
		return &contentStream, nil
	}

	return nil, twirp.NotFoundError(fmt.Sprintf("Stream not found for customerId: %s, contentId: %s", customerId, contentId))
}

// ListStreams returns streams for a customer id in usher-style response
func (c *InMemoryCache) ListStreams(ctx context.Context, customerId string) ([]usher.UsherStreamResponse, error) {
	customerStreams, err := c.get(ctx, customerId)
	if err != nil {
		return nil, err
	}

	response := []usher.UsherStreamResponse{}
	for _, stream := range customerStreams {
		response = append(response, stream)
	}

	return response, nil
}

func (c *InMemoryCache) get(ctx context.Context, customerId string) (cacheValue, error) {
	value, found := c.cacheImpl.Get(customerId)
	if c.statter != nil {
		_ = c.statter.Inc("cache_get", 1, 1.0)
	}
	if !found {
		if c.statter != nil {
			_ = c.statter.Inc("cache_miss", 1, 1.0)
		}
		logging.Debug(ctx, "Cache miss for Customer ID %s", customerId)
		err := c.setStreams(ctx, customerId)
		if err != nil {
			return nil, err
		}
		value, _ = c.cacheImpl.Get(customerId)
	}

	// return stream for contentId and customerId
	customerStreams, ok := value.(cacheValue)
	if !ok {
		return nil, twirp.InternalError("invalid cached value type")
	}
	return customerStreams, nil
}

func (c *InMemoryCache) setStreams(ctx context.Context, customerId string) error {
	// populate the cache by calling Usher
	stl, err := c.usher.ListStreams(customerId)
	if err != nil {
		if strings.Contains(err.Error(), "404") {
			return twirp.NotFoundError(fmt.Sprintf("Stream not found for customerId: %s ", customerId))
		}
		return twirp.InternalErrorWith(err)
	}

	value := make(cacheValue)
	for _, stream := range stl {
		value[stream.ContentId] = stream
	}

	c.cacheImpl.Set(customerId, value, c.defaultExpiration)
	return nil
}
