package caching

import (
	"context"
	"fmt"
	"testing"

	"github.com/stretchr/testify/require"

	"code.justin.tv/video/lvsapi/internal/caching/gocache/gocachefakes"
	"code.justin.tv/video/lvsapi/internal/usher"
	"code.justin.tv/video/lvsapi/internal/usher/usherfakes"
)

func TestGetStream(t *testing.T) {
	// Basic test to check the response when cache has the required data
	value := make(cacheValue)
	value["test-content"] = usher.UsherStreamResponse{
		ContentId: "test-content",
	}

	cacheImpl := gocachefakes.FakeGoCache{}
	cacheImpl.GetReturns(value, true)

	localCache := &InMemoryCache{
		cacheImpl: &cacheImpl,
		usher:     &usherfakes.FakeUsherBackend{},
	}

	resp, err := localCache.GetStream(context.Background(), "test-customer", "test-content")
	require.Nil(t, err)
	require.NotNil(t, resp)
	require.Equal(t, "test-content", resp.ContentId)
}

func TestGetStreamCacheCustomerNotFound(t *testing.T) {
	// When the gocache does not have the data for the customer,
	// we expect to get the data from Usher backend

	// set mock cache to return cache miss on first call
	cacheImpl := gocachefakes.FakeGoCache{}
	cacheImpl.GetReturnsOnCall(0, nil, false)

	value := make(cacheValue)
	value["test-content-1"] = usher.UsherStreamResponse{ContentId: "test-content-1"}
	cacheImpl.GetReturnsOnCall(1, value, true)

	// set mock usher backend to return some streams
	usherBackend := usherfakes.FakeUsherBackend{}
	usherBackend.ListStreamsReturns([]usher.UsherStreamResponse{
		usher.UsherStreamResponse{ContentId: "test-content-1"},
		usher.UsherStreamResponse{ContentId: "test-content-2"},
	}, nil)

	localCache := &InMemoryCache{
		cacheImpl: &cacheImpl,
		usher:     &usherBackend,
	}

	resp, err := localCache.GetStream(context.Background(), "test-customer", "test-content-1")

	// make sure the response is as expected
	require.Nil(t, err)
	require.NotNil(t, resp)
	require.Equal(t, "test-content-1", resp.ContentId)
	require.Equal(t, 1, usherBackend.ListStreamsCallCount())

	// Check call args
	argCustomer, argValue, _ := cacheImpl.SetArgsForCall(0)
	require.Equal(t, "test-customer", argCustomer)
	contentIds := []string{}
	for id := range argValue.(cacheValue) {
		contentIds = append(contentIds, id)
	}
	require.Contains(t, contentIds, "test-content-1")
	require.Contains(t, contentIds, "test-content-2")

}

func TestGetStreamCacheContentNotFound(t *testing.T) {
	// When the gocache does not have the data for the contentId,
	// but has data for requested customerid, we dont expect
	// usher backend to be called

	// set gocache mock to return certain contentid
	value := make(cacheValue)
	value["test-content-2"] = usher.UsherStreamResponse{ContentId: "test-content-2"}
	cacheImpl := gocachefakes.FakeGoCache{}
	cacheImpl.GetReturns(value, true)

	usherBackend := usherfakes.FakeUsherBackend{}
	localCache := &InMemoryCache{
		cacheImpl: &cacheImpl,
		usher:     &usherBackend,
	}

	// make a request for a different contentId
	resp, err := localCache.GetStream(context.Background(), "test-customer", "test-content-1")

	// make sure we get a 404 back
	require.Nil(t, resp)
	require.Contains(t, err.Error(), "not_found")

	// make sure usher backend was not called
	require.Equal(t, 0, usherBackend.ListStreamsCallCount())
}

func TestGetStreamUsherCustomerNotFound(t *testing.T) {
	// When gocache doesn't have data for a customerId,
	// and Usher also returns 404, make sure we return 404

	// set gocache mock to return nil
	cacheImpl := gocachefakes.FakeGoCache{}
	cacheImpl.GetReturns(nil, false)

	// set usher backend to return 404
	usherBackend := usherfakes.FakeUsherBackend{}
	usherBackend.ListStreamsReturns(nil, fmt.Errorf("404"))

	localCache := &InMemoryCache{
		cacheImpl: &cacheImpl,
		usher:     &usherBackend,
	}

	// make a request
	resp, err := localCache.GetStream(context.Background(), "test-customer", "test-content")

	// make sure we get a 404 back
	require.Nil(t, resp)
	require.Contains(t, err.Error(), "not_found")

	// make sure usher backend was called
	require.Equal(t, 1, usherBackend.ListStreamsCallCount())
}

func TestGetStreamUsherContentNotFound(t *testing.T) {
	// When gocache doesn't have data and usher response does not have
	// data for requested contentId, make sure we return 404

	// set gocache mock to return nil
	cacheImpl := gocachefakes.FakeGoCache{}
	cacheImpl.GetReturnsOnCall(0, nil, false)

	value := make(cacheValue)
	value["test-content-1"] = usher.UsherStreamResponse{ContentId: "test-content-1"}
	cacheImpl.GetReturnsOnCall(1, value, true)

	usherBackend := usherfakes.FakeUsherBackend{}
	usherBackend.ListStreamsReturns([]usher.UsherStreamResponse{
		usher.UsherStreamResponse{ContentId: "test-content-1"},
	}, nil)

	localCache := &InMemoryCache{
		cacheImpl: &cacheImpl,
		usher:     &usherBackend,
	}

	// make a request
	resp, err := localCache.GetStream(context.Background(), "test-customer", "test-content-2")

	// make sure we get a 404 back
	require.Nil(t, resp)
	require.Contains(t, err.Error(), "not_found")

	// make sure usher backend was called
	require.Equal(t, 1, usherBackend.ListStreamsCallCount())
}

func TestGetStreamUsherError(t *testing.T) {
	// When gocache doesn't have the data and usher returns an error
	// make sure we return InternalError back

	// set gocache mock to return nil
	cacheImpl := gocachefakes.FakeGoCache{}
	cacheImpl.GetReturns(nil, false)

	usherBackend := usherfakes.FakeUsherBackend{}
	usherBackend.ListStreamsReturns(nil, fmt.Errorf("test-error"))

	localCache := &InMemoryCache{
		cacheImpl: &cacheImpl,
		usher:     &usherBackend,
	}

	// make a request
	resp, err := localCache.GetStream(context.Background(), "test-customer", "test-content")

	// make sure we return internal error back
	require.Nil(t, resp)
	require.Contains(t, err.Error(), "internal")
}

func TestListStreams(t *testing.T) {
	// Basic test to check the response when cache has the required data
	value := make(cacheValue)
	value["test-content-1"] = usher.UsherStreamResponse{
		ContentId: "test-content-1",
	}
	value["test-content-2"] = usher.UsherStreamResponse{
		ContentId: "test-content-2",
	}

	cacheImpl := gocachefakes.FakeGoCache{}
	cacheImpl.GetReturns(value, true)

	localCache := &InMemoryCache{
		cacheImpl: &cacheImpl,
		usher:     &usherfakes.FakeUsherBackend{},
	}

	resp, err := localCache.ListStreams(context.Background(), "test-customer")
	require.Nil(t, err)
	require.Equal(t, 2, len(resp))

	contentIds := []string{}
	for _, stream := range resp {
		contentIds = append(contentIds, stream.ContentId)
	}
	require.Contains(t, contentIds, "test-content-1")
	require.Contains(t, contentIds, "test-content-2")
}

func TestListStreamsCacheCustomerNotFound(t *testing.T) {
	// When gocache doesn't have the data, make sure
	// we call usher

	cacheImpl := gocachefakes.FakeGoCache{}
	cacheImpl.GetReturnsOnCall(0, nil, false)

	value := make(cacheValue)
	value["test-content"] = usher.UsherStreamResponse{
		ContentId: "test-content",
	}
	cacheImpl.GetReturnsOnCall(1, value, true)

	usherBackend := usherfakes.FakeUsherBackend{}
	localCache := &InMemoryCache{
		cacheImpl: &cacheImpl,
		usher:     &usherBackend,
	}

	resp, err := localCache.ListStreams(context.Background(), "test-customer")
	require.Nil(t, err)
	require.Equal(t, 1, len(resp))
	require.Equal(t, "test-content", resp[0].ContentId)

	// Make sure usher was called
	require.Equal(t, 1, usherBackend.ListStreamsCallCount())
}

func TestListStreamsUsherCustomerNotFound(t *testing.T) {
	// When gocache doesn't have the data, and usher also
	// doesn't have the data, make sure we return 404

	cacheImpl := gocachefakes.FakeGoCache{}
	cacheImpl.GetReturnsOnCall(0, nil, false)

	usherBackend := usherfakes.FakeUsherBackend{}
	usherBackend.ListStreamsReturns(nil, fmt.Errorf("404"))

	localCache := &InMemoryCache{
		cacheImpl: &cacheImpl,
		usher:     &usherBackend,
	}

	resp, err := localCache.ListStreams(context.Background(), "test-customer")

	require.NotNil(t, err)
	require.Nil(t, resp)

	require.Contains(t, err.Error(), "not_found")
}

func TestListStreamsUsherError(t *testing.T) {
	// When gocache doesn't have the data, and usher returns an error
	// make sure we return an internal error

	cacheImpl := gocachefakes.FakeGoCache{}
	cacheImpl.GetReturnsOnCall(0, nil, false)

	usherBackend := usherfakes.FakeUsherBackend{}
	usherBackend.ListStreamsReturns(nil, fmt.Errorf("test-error"))

	localCache := &InMemoryCache{
		cacheImpl: &cacheImpl,
		usher:     &usherBackend,
	}

	resp, err := localCache.ListStreams(context.Background(), "test-customer")

	require.NotNil(t, err)
	require.Nil(t, resp)

	require.Contains(t, err.Error(), "internal")
}
