package digestion

import (
	"context"
	"fmt"
	"net/http"
	"strings"
	"time"

	"code.justin.tv/video/digestion/pkg/dgn"
	"github.com/twitchtv/twirp"
)

const digestionTimeout = time.Second * 1

//DigestionAPI defines the client which can be used to interact with Digestion
type DigestionAPI struct {
	clients []dgn.DigestionV1
}

//NewDigestionAPI returns the client which can be used to query Digestion
func NewDigestionAPI(commaSeparatedHosts string) *DigestionAPI {
	var clients []dgn.DigestionV1
	for _, host := range strings.Split(commaSeparatedHosts, ",") {
		clients = append(clients, dgn.NewDigestionV1ProtobufClient(host, &http.Client{
			Timeout: digestionTimeout,
		}))
	}

	return &DigestionAPI{
		clients: clients,
	}
}

//DeleteStream is responsible for calling the delete stream api on digestion which also disconnects rtmp stream
func (d *DigestionAPI) DeleteStream(ctx context.Context, channel string) error {
	if len(d.clients) == 0 {
		return fmt.Errorf("no digestion clients")
	}

	var errors []string
	for _, client := range d.clients {
		_, err := client.DisconnectChannel(ctx, &dgn.ChannelRequest{
			Channel: channel,
		})
		if err == nil {
			// One success is sufficient
			return nil
		}
		errors = append(errors, processError(err))
	}

	return fmt.Errorf("errors occurred: %s", strings.Join(errors, ", "))
}

const defaultRateLimitPerSecond = 5

// AddMetadata inserts metadata into a live stream. The rate of insertion operations
// that can be performed per-channel is limited, and may be increased for specific customers
// in the future. At the time of writing, the maximum possible limit is 20/sec (enforced by digestion)
func (d *DigestionAPI) AddMetadata(ctx context.Context, channel, metadata string) error {
	if len(d.clients) == 0 {
		return fmt.Errorf("no digestion clients")
	}
	var errors []string
	req := dgn.PutIngestMetadataRequest{
		Channel:     channel,
		Metadata:    metadata,
		RequestedAt: time.Now().Format(time.RFC3339),
		RateLimit:   defaultRateLimitPerSecond,
	}
	for _, client := range d.clients {
		_, err := client.PutIngestMetadata(ctx, &req)
		if err == nil {
			// One success is sufficient
			return nil
		}
		errors = append(errors, processError(err))
	}

	return fmt.Errorf("errors occurred: %s", strings.Join(errors, ", "))
}

func processError(err error) string {
	twerr, ok := err.(twirp.Error)
	if !ok {
		twerr = twirp.InternalErrorWith(err)
	}
	return fmt.Sprintf("%s %d: %s", twerr.Code(), twirp.ServerHTTPStatusFromErrorCode(twerr.Code()), twerr.Msg())
}
