package digestion

import (
	"testing"

	"strings"

	"context"

	"code.justin.tv/video/digestion/pkg/dgn"
	"code.justin.tv/video/digestion/pkg/dgn/dgnfakes"
	"github.com/stretchr/testify/require"
	"github.com/twitchtv/twirp"
)

func TestAllDigestionDeleteSuccess(t *testing.T) {
	mockDigestionClient1 := dgnfakes.FakeDigestionV1{}
	mockDigestionClient2 := dgnfakes.FakeDigestionV1{}
	mockDigestionClient1.DisconnectChannelReturns(&dgn.Channel{}, nil)
	mockDigestionClient2.DisconnectChannelReturns(&dgn.Channel{}, nil)
	dgnAPI := DigestionAPI{
		clients: []dgn.DigestionV1{&mockDigestionClient1, &mockDigestionClient2},
	}
	err := dgnAPI.DeleteStream(context.Background(), "channel")
	require.Nil(t, err)
}

func TestSomeDigestionDeleteFailure(t *testing.T) {
	mockDigestionClient1 := dgnfakes.FakeDigestionV1{}
	mockDigestionClient2 := dgnfakes.FakeDigestionV1{}
	mockDigestionClient1.DisconnectChannelReturns(&dgn.Channel{}, nil)
	mockDigestionClient2.DisconnectChannelReturns(nil, twirp.NotFoundError("not found"))
	dgnAPI := DigestionAPI{
		clients: []dgn.DigestionV1{&mockDigestionClient1, &mockDigestionClient2},
	}
	err := dgnAPI.DeleteStream(context.Background(), "channel")
	require.Nil(t, err)
}

func TestAllDigestionDeleteFailures(t *testing.T) {
	mockDigestionClient1 := dgnfakes.FakeDigestionV1{}
	mockDigestionClient2 := dgnfakes.FakeDigestionV1{}
	mockDigestionClient1.DisconnectChannelReturns(nil, twirp.NotFoundError("not found"))
	mockDigestionClient2.DisconnectChannelReturns(nil, twirp.NotFoundError("not found"))
	dgnAPI := DigestionAPI{
		clients: []dgn.DigestionV1{&mockDigestionClient1, &mockDigestionClient2},
	}
	err := dgnAPI.DeleteStream(context.Background(), "channel")
	require.NotNil(t, err)
	require.True(t, strings.Contains(err.Error(), "404"))
}

func TestAllDigestionAddMetadataSuccess(t *testing.T) {
	mockDigestionClient1 := dgnfakes.FakeDigestionV1{}
	mockDigestionClient2 := dgnfakes.FakeDigestionV1{}
	mockDigestionClient1.PutIngestMetadataReturns(&dgn.PutIngestMetadataResponse{}, nil)
	mockDigestionClient2.PutIngestMetadataReturns(&dgn.PutIngestMetadataResponse{}, nil)
	dgnAPI := DigestionAPI{
		clients: []dgn.DigestionV1{&mockDigestionClient1, &mockDigestionClient2},
	}
	err := dgnAPI.AddMetadata(context.Background(), "channel", "metadata")
	require.Nil(t, err)
}

func TestSomeDigestionAddMetadataFailure(t *testing.T) {
	mockDigestionClient1 := dgnfakes.FakeDigestionV1{}
	mockDigestionClient2 := dgnfakes.FakeDigestionV1{}
	mockDigestionClient1.PutIngestMetadataReturns(&dgn.PutIngestMetadataResponse{}, nil)
	mockDigestionClient2.PutIngestMetadataReturns(nil, twirp.NotFoundError("not found"))
	dgnAPI := DigestionAPI{
		clients: []dgn.DigestionV1{&mockDigestionClient1, &mockDigestionClient2},
	}
	err := dgnAPI.AddMetadata(context.Background(), "channel", "metadata")
	require.Nil(t, err)
}

func TestAllDigestionAddMetadataFailures(t *testing.T) {
	mockDigestionClient1 := dgnfakes.FakeDigestionV1{}
	mockDigestionClient2 := dgnfakes.FakeDigestionV1{}
	mockDigestionClient1.PutIngestMetadataReturns(nil, twirp.NotFoundError("not found"))
	mockDigestionClient2.PutIngestMetadataReturns(nil, twirp.NotFoundError("not found"))
	dgnAPI := DigestionAPI{
		clients: []dgn.DigestionV1{&mockDigestionClient1, &mockDigestionClient2},
	}
	err := dgnAPI.AddMetadata(context.Background(), "channel", "metadata")
	require.NotNil(t, err)
	require.True(t, strings.Contains(err.Error(), "404"))
}
