package logging

import (
	"bufio"
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"log"
	"net/http"
	"net/http/httptest"
	"testing"

	"goji.io/pat"

	"code.justin.tv/common/chitin"
	"code.justin.tv/release/trace"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"github.com/twitchtv/twirp"

	netcontext "context"

	twirpexample "github.com/twitchtv/twirp/example"

	goji "goji.io"
)

func TestHttpLogger(t *testing.T) {
	buf := &bytes.Buffer{}
	l := New(Config{
		Out: buf,
	})

	var tid *trace.ID
	tidLogger := func(h http.Handler) http.Handler {
		return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			tid = chitin.GetTraceID(r.Context())
			h.ServeHTTP(w, r)
		})
	}

	mux := goji.NewMux()
	mux.Use(Middleware(l))
	mux.Use(tidLogger)
	s := httptest.NewServer(chitin.Handler(mux))

	resp, err := http.Get(s.URL + "/test/path")
	require.NoError(t, err)
	err = resp.Body.Close()
	require.NoError(t, err)
	s.Close()

	var logLine map[string]interface{}
	err = json.NewDecoder(buf).Decode(&logLine)
	require.NoError(t, err)

	fields := Fields{
		"method":           http.MethodGet,
		"path":             "/test/path",
		"status":           resp.StatusCode,
		"duration":         nil,
		"headers-duration": nil,
		"trace-id":         tid.String(),
	}

	for k, v := range fields {
		assert.Contains(t, logLine, k)
		if v != nil {
			assert.EqualValues(t, logLine[k], v)
		}
	}
}

func TestHttpLoggerNoWrite(t *testing.T) {
	buf := &bytes.Buffer{}
	l := New(Config{
		Out: buf,
	})

	mux := goji.NewMux()
	mux.Use(Middleware(l))
	mux.Handle(pat.Get("/*"), http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {}))
	s := httptest.NewServer(mux)

	resp, err := http.Get(s.URL)
	require.NoError(t, err)
	err = resp.Body.Close()
	require.NoError(t, err)
	s.Close()

	var logLine map[string]interface{}
	err = json.NewDecoder(buf).Decode(&logLine)
	require.NoError(t, err)

	assert.EqualValues(t, http.StatusOK, logLine["status"])
}

func TestHttpLoggerNoStatus(t *testing.T) {
	buf := &bytes.Buffer{}
	l := New(Config{
		Out: buf,
	})

	mux := goji.NewMux()
	mux.Use(Middleware(l))
	mux.Handle(pat.Get("/*"), http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) { w.Write(nil) }))
	s := httptest.NewServer(mux)

	resp, err := http.Get(s.URL)
	require.NoError(t, err)
	err = resp.Body.Close()
	require.NoError(t, err)
	s.Close()

	var logLine map[string]interface{}
	err = json.NewDecoder(buf).Decode(&logLine)
	require.NoError(t, err)

	assert.EqualValues(t, http.StatusOK, logLine["status"])
}

// just validating that HandlerError can be called on a background context
// and nothing happens (like a panic for example)
func TestHandlerErrorNotInHandler(t *testing.T) {
	HandlerError(context.Background(), errors.New("test"))
}

func TestHandlerError(t *testing.T) {
	buf := &bytes.Buffer{}
	l := New(Config{
		Out: buf,
	})

	testError := errors.New("test error")
	errSetter := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		HandlerError(r.Context(), testError)
	})

	mux := goji.NewMux()
	mux.Use(Middleware(l))
	mux.Handle(pat.Get("/seterror"), errSetter)
	s := httptest.NewServer(mux)

	resp, err := http.Get(s.URL + "/seterror")
	require.NoError(t, err)
	err = resp.Body.Close()
	require.NoError(t, err)

	resp, err = http.Get(s.URL + "/notseterror")
	require.NoError(t, err)
	err = resp.Body.Close()
	require.NoError(t, err)

	s.Close()

	log.Printf("%q", string(buf.Bytes()))

	var logLines []map[string]interface{}

	scanner := bufio.NewScanner(buf)
	for scanner.Scan() {
		var line map[string]interface{}
		require.NoError(t, json.Unmarshal([]byte(scanner.Text()), &line))
		logLines = append(logLines, line)
	}

	require.Len(t, logLines, 2)
	assert.Equal(t, testError.Error(), logLines[0]["error"])
	assert.NotContains(t, "error", logLines[1])
}

func TestHandlerErrorHTTPAbort(t *testing.T) {
	buf := &bytes.Buffer{}
	l := New(Config{
		Out: buf,
	})

	testError := errors.New("test error")
	aborter := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		HandlerError(r.Context(), testError)
		panic(http.ErrAbortHandler)
	})

	mux := goji.NewMux()
	mux.Use(Middleware(l))
	mux.Handle(pat.Get("/*"), aborter)
	s := httptest.NewServer(mux)

	_, err := http.Get(s.URL)
	require.Error(t, err)
	s.Close()

	var logLine map[string]interface{}
	err = json.NewDecoder(buf).Decode(&logLine)
	require.NoError(t, err)

	assert.Equal(t, logLine["error"], testError.Error())
}

func TestMultipleHandlerError(t *testing.T) {
	buf := &bytes.Buffer{}
	l := New(Config{
		Out: buf,
	})

	testError := errors.New("test error")
	errSetter := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		HandlerError(r.Context(), testError)
		HandlerError(r.Context(), testError)
	})

	mux := goji.NewMux()
	mux.Use(Middleware(l))
	mux.Handle(pat.Get("/*"), errSetter)
	s := httptest.NewServer(mux)

	resp, err := http.Get(s.URL)
	require.NoError(t, err)
	err = resp.Body.Close()
	require.NoError(t, err)
	s.Close()

	var logLine map[string]interface{}
	err = json.NewDecoder(buf).Decode(&logLine)
	require.NoError(t, err)

	assert.Equal(t, logLine["error"], flattenErrors([]error{testError, testError}).Error())
}

type errorHaberdasher struct {
	err error
}

func (eh *errorHaberdasher) MakeHat(ctx netcontext.Context, size *twirpexample.Size) (*twirpexample.Hat, error) {
	return nil, eh.err
}

func TestTwirpHooks(t *testing.T) {
	buf := &bytes.Buffer{}
	l := New(Config{
		Out: buf,
	})

	testError := errors.New("test error")
	handler := twirpexample.NewHaberdasherServer(&errorHaberdasher{testError}, TwirpHooks())

	mux := goji.NewMux()
	mux.Use(Middleware(l))
	mux.Handle(pat.Post("/*"), handler)
	s := httptest.NewServer(mux)
	defer s.Close()

	client := twirpexample.NewHaberdasherProtobufClient(s.URL, &http.Client{})
	_, err := client.MakeHat(context.Background(), &twirpexample.Size{})
	assert.Equal(t, err.(twirp.Error).Msg(), testError.Error())

	var logLine map[string]interface{}
	err = json.NewDecoder(buf).Decode(&logLine)
	require.NoError(t, err)

	assert.Equal(t, logLine["error"], twirp.NewError(twirp.Internal, testError.Error()).Error())
}
