package metrics

import (
	"fmt"
	"log"
	"strings"
	"sync"
	"sync/atomic"
	"time"

	"code.justin.tv/commerce/AmazonMWSGoClient/mws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/cenkalti/backoff"
)

// Possible units that we support
// TODO: The units declared here should eventually be added to the client library
const (
	UnitCounter     = "Count"
	UnitBytes       = "Bytes"
	UnitMillisecond = "Milliseconds"

	marketPlace = "us-west-2"

	// We can buffer only this many metrics before we start dropping them
	bufferSize = 32
)

// This should only be changed for testing purposes
// NOTE: This approach works if there is no state kept in this struct.
var defaultMWSClient = mws.IAmazonMWSGoClient(mws.AmazonMWSGoClient{})

// This allows us to supply a customer ticker for testing
var defaultStatsTicker = func() *time.Ticker {
	return time.NewTicker(time.Minute)
}

// Client is an instance of the metric client
type Client struct {
	creds        *credentials.Credentials
	serviceName  string
	hostname     string
	hostSpecific bool // This is specific to the client we are using
	isProdMetric bool // This is specific to the client we are using
	env          string

	// Queue of metrics that are about to be reported
	queue chan metric
	wg    sync.WaitGroup

	statsTicker *time.Ticker
	stop        chan struct{}

	// Record metric on how well the metrics system is doing
	queueCallCount  uint64 // Number of calls made to the Record method
	flushCallCount  uint64 // Number of times we flushed the metrics to the server
	queueDropCount  uint64 // Failed to enqueue due to buffer full
	recordDropCount uint64 // Failed to report to the metrics server

}

// The metric record details that would be queued up
type metric struct {
	clientID string
	method   string
	name     string
	unit     string
	value    float64
}

// New returns a new metrics client
// serviceName - the name of the service from which the metrics will be originating
// hostname - the hostname from which the metrics will be originating
func New(serviceName, hostname, env string, creds *credentials.Credentials) *Client {
	env = strings.ToLower(env)
	return &Client{
		creds:        creds,
		serviceName:  serviceName,
		hostname:     hostname,
		hostSpecific: hostname != "",
		isProdMetric: env == "prod" || env == "production" || env == "production-updated",
		env:          env,
		queue:        make(chan metric, bufferSize),
		wg:           sync.WaitGroup{},
		statsTicker:  defaultStatsTicker(),
		stop:         make(chan struct{}),
	}
}

// Start a background routine to record the stats that are queued up
// TODO: Consider batching and/or using multiple workers in the future
func (c *Client) Start() {
	// Report internal library stats
	c.wg.Add(1)
	go func() {
		defer c.wg.Done()
		for {
			select {
			case <-c.statsTicker.C:
				queueCallCount := atomic.SwapUint64(&c.queueCallCount, 0)
				flushCallCount := atomic.SwapUint64(&c.flushCallCount, 0)
				queueDropCount := atomic.SwapUint64(&c.queueDropCount, 0)
				recordDropCount := atomic.SwapUint64(&c.recordDropCount, 0)

				c.recordUntilSuccess("", "metrics", "QueueCallCount", UnitCounter, float64(queueCallCount))
				c.recordUntilSuccess("", "metrics", "FlushCallCount", UnitCounter, float64(flushCallCount))
				c.recordUntilSuccess("", "metrics", "QueueDropCount", UnitCounter, float64(queueDropCount))
				c.recordUntilSuccess("", "metrics", "RecordDropCount", UnitCounter, float64(recordDropCount))
			case <-c.stop:
				return
			}
		}
	}()

	// Record metrics from the rest of the application
	c.wg.Add(1)
	go func() {
		defer c.wg.Done()
		t := 0
		for m := range c.queue {
			t++
			atomic.AddUint64(&c.flushCallCount, 1)
			err := c.record(m.clientID, m.method, m.name, m.unit, m.value)
			if err != nil {
				atomic.AddUint64(&c.recordDropCount, 1)
				log.Printf("Failed to record metric: %v, %v", m, err)
			}
		}
	}()
}

// Stop the client. It waits for the client to record everything that was queued up
// before returning.
// NOTE: calling Record after stop will cause panic
func (c *Client) Stop() {
	close(c.queue)
	c.statsTicker.Stop()
	close(c.stop)
	c.wg.Wait()
}

// Record a metric with the given parameters
// clientID - The ID of the client for which we are recording this metric
// method - The method for which we are recording the stat
// name - The name of the metric
// unit - The unit that will be used to record this metric
// NOTE: calling record after the client has been stopped will cause panic
func (c *Client) Record(clientID, method, name, unit string, value float64) error {
	m := metric{clientID, method, name, unit, value}
	atomic.AddUint64(&c.queueCallCount, 1)
	select {
	case c.queue <- m:
		return nil
	default:
		atomic.AddUint64(&c.queueDropCount, 1)
		return fmt.Errorf("Failed to queue up the metric")
	}
}

// Record a single metric
func (c *Client) record(clientID, method, name, unit string, value float64) error {
	// Initialize metric object.
	metric := mws.NewMetric(c.isProdMetric, c.hostname, c.hostSpecific, marketPlace, c.serviceName, method, clientID, name)

	// Add the actual values of the metric.
	metric.AddValue(value)
	metric.Unit = unit
	metric.Timestamp = time.Now().UTC()

	// Metric report metadata.
	minPeriod := mws.PeriodOneMinute
	maxPeriod := mws.PeriodOneHour

	// Initialize MetricReport and add the Metric.
	metricReport := mws.NewMetricReport(c.isProdMetric, c.hostname, minPeriod, maxPeriod, "", c.env, clientID)
	metricReport.AddMetric(metric)

	// Wrap the MetricReport in a request object.
	req := mws.NewPutMetricsForAggregationRequest()
	req.AddMetricReport(metricReport)

	// Select an endpoint. Make sure the region matches the credentials.
	// TODO: This needs to be improved to select region accordingly
	region := mws.Regions["CMH"] // CMH is us-west-2.

	// Make the call to log the metric
	_, err := defaultMWSClient.PutMetricsForAggregation(req, region, c.creds)
	return err
}

// This is the same as record but it only returns after the metrics were successfully recorded
func (c *Client) recordUntilSuccess(clientID, method, name, unit string, value float64) {
	exp := backoff.NewExponentialBackOff()
	exp.MaxElapsedTime = 0 // retry until success

	operation := func() error {
		return c.record(clientID, method, name, unit, value)
	}

	err := backoff.Retry(operation, exp)
	if err != nil {
		log.Printf("recordUntilSuccess should never fail: %v\n", err)
	}
}
