package service

import (
	"context"
	"net/http"
	"net/http/httptest"
	"testing"
	"time"

	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/golang/protobuf/proto"
	"github.com/golang/protobuf/ptypes"
	"github.com/stretchr/testify/require"

	"fmt"

	"strings"

	"encoding/base64"

	"code.justin.tv/video/lvsapi/internal/auth"
	"code.justin.tv/video/lvsapi/internal/awsutils/awsutilsfakes"
	"code.justin.tv/video/lvsapi/internal/caching/cachingfakes"
	"code.justin.tv/video/lvsapi/internal/digestion/digestionfakes"
	"code.justin.tv/video/lvsapi/internal/metrics"
	"code.justin.tv/video/lvsapi/internal/usher"
	"code.justin.tv/video/lvsapi/internal/viewcounts/viewcountsfakes"
	"code.justin.tv/video/lvsapi/rpc/lvs"
	"code.justin.tv/video/lvsapi/streamkey"
	"github.com/golang/protobuf/ptypes/wrappers"
	"github.com/twitchtv/twirp"
)

func TestValidContentId(t *testing.T) {
	// positive cases
	require.True(t, IsValidContentID("contentID"))
	require.True(t, IsValidContentID("1"))
	require.True(t, IsValidContentID("a"))
	require.True(t, IsValidContentID("a1b3"))
	require.True(t, IsValidContentID("twitchmedia_2"))
	require.True(t, IsValidContentID("twitchmedia_2-1"))
	require.True(t, IsValidContentID("f9d3952b-5915-474f-8c5f-38aeb9332626"))

	// Negative cases
	require.False(t, IsValidContentID(""))
	require.False(t, IsValidContentID("@#$@#$"))
	require.False(t, IsValidContentID("aaa aaa"))
	require.False(t, IsValidContentID("aaaa$aaaa"))
	require.False(t, IsValidContentID("1234562323232323232789123456789123456789123456789123456789123456789123456789"))
}

type mockServiceConfig struct {
	customerID   string
	secret       *streamkey.Secret
	secretSource streamkey.SecretSource
}

func (m *mockServiceConfig) streamKey(ttl time.Duration) (string, error) {
	ts, err := ptypes.TimestampProto(time.Now().Add(ttl))
	if err != nil {
		return "", err
	}

	sk := streamkey.NewV1(m.customerID, &streamkey.PrivateData{ExpirationTime: ts})
	return streamkey.Encrypt(context.Background(), m.secretSource, sk)
}

func (m *mockServiceConfig) service(cache *cachingfakes.FakeCacheInterface, s3APIS *awsutilsfakes.FakeS3APIs, dgnAPI *digestionfakes.FakeDigestionBackend, viewcountsAPI *viewcountsfakes.FakeViewcountsCache) http.Handler {
	creds := credentials.NewStaticCredentials("testId", "testSecret", "")
	metrics := metrics.New("lvsapi-test", "", "", creds)
	return auth.FromHeader(lvs.NewLiveVideoServiceServer(New(m.secretSource, cache, s3APIS, metrics, dgnAPI, viewcountsAPI), nil))
}

func (m *mockServiceConfig) internalService() http.Handler {
	return lvs.NewLiveVideoServiceInternalServer(NewInternal(m.secretSource), nil)
}

func mockService() (*mockServiceConfig, error) {
	customerID := "test-customer"
	secret, err := streamkey.GenerateSecret()
	if err != nil {
		return nil, err
	}
	secretSource := make(streamkey.MapSecretSource)
	if err = secretSource.Set(context.Background(), customerID, secret); err != nil {
		return nil, err
	}

	return &mockServiceConfig{
		customerID:   customerID,
		secret:       secret,
		secretSource: secretSource,
	}, nil
}

func TestCreateStreamKey(t *testing.T) {
	svc, err := mockService()
	require.NoError(t, err)

	s3APIS := awsutilsfakes.FakeS3APIs{}
	s3APIS.GetBucketRegionReturns("us-west-2", nil)
	server := httptest.NewServer(svc.service(&cachingfakes.FakeCacheInterface{}, &s3APIS, &digestionfakes.FakeDigestionBackend{}, &viewcountsfakes.FakeViewcountsCache{}))
	defer server.Close()

	client := lvs.NewLiveVideoServiceProtobufClient(server.URL, http.DefaultClient)
	header := make(http.Header)
	header.Set(auth.ForceAuthHeader, svc.customerID)
	ctx, err := twirp.WithHTTPRequestHeaders(context.Background(), header)
	require.NoError(t, err)

	resp, err := client.CreateStreamKey(ctx, &lvs.CreateStreamKeyRequest{
		ContentId:               "test",
		S3Bucket:                "testbucket",
		S3Prefix:                "testprefix",
		SnsNotificationEndpoint: "arn:aws:sns:us-west-2:123456:test",
		TtlSeconds:              10,
		EnableLowLatency:        true,
		CdnUrl:                  "http://test.cloudfront.net",
	})

	require.NoError(t, err)
	require.NotNil(t, resp)
	require.Equal(t, resp.ContentId, "test")
	require.Equal(t, resp.CustomerId, "test-customer")
	require.Equal(t, resp.StreamkeyMetadata.SnsNotificationEndpoint, "arn:aws:sns:us-west-2:123456:test")
	require.Equal(t, resp.StreamkeyMetadata.S3Bucket, "testbucket")
	require.Equal(t, resp.StreamkeyMetadata.S3Prefix, "testprefix")
	require.Equal(t, resp.StreamkeyMetadata.EnableLowLatency, true)
	require.Equal(t, resp.PlaybackUrl,
		"https://usher.ttvnw.net/api/lvs/hls/lvs.test-customer.test.m3u8?allow_source=true&player_backend=mediaplayer")
	require.Equal(t, resp.StreamkeyMetadata.CdnUrl, "http://test.cloudfront.net")

	//Test the created stream key expiration time

	expirationTime, err := ptypes.Timestamp(resp.StreamkeyMetadata.ExpirationTime)
	require.Nil(t, err)

	currentTime := time.Now()

	timeToExpire := expirationTime.Sub(currentTime).Seconds()

	// Make sure the expiration time is not more than 10 seconds and greater than 0
	require.True(t, timeToExpire > 0)
	require.True(t, timeToExpire <= 10)

}

func TestCreateStreamKeyInvalidSNS(t *testing.T) {
	svc, err := mockService()
	require.NoError(t, err)

	s3APIS := awsutilsfakes.FakeS3APIs{}
	s3APIS.GetBucketRegionReturns("us-west-2", nil)
	server := httptest.NewServer(svc.service(&cachingfakes.FakeCacheInterface{}, &s3APIS, &digestionfakes.FakeDigestionBackend{}, &viewcountsfakes.FakeViewcountsCache{}))
	defer server.Close()

	client := lvs.NewLiveVideoServiceProtobufClient(server.URL, http.DefaultClient)
	header := make(http.Header)
	header.Set(auth.ForceAuthHeader, svc.customerID)
	ctx, err := twirp.WithHTTPRequestHeaders(context.Background(), header)
	require.NoError(t, err)

	_, err = client.CreateStreamKey(ctx, &lvs.CreateStreamKeyRequest{
		ContentId:               "test",
		SnsNotificationEndpoint: "invalid_not_arn",
	})

	require.Error(t, err)
	require.True(t, strings.Contains(err.Error(), "invalid"))

}

func TestCreateStreamKeyInvalidRegion(t *testing.T) {
	svc, err := mockService()
	require.NoError(t, err)

	s3APIS := awsutilsfakes.FakeS3APIs{}
	s3APIS.GetBucketRegionReturns("us-east-2", nil)
	server := httptest.NewServer(svc.service(&cachingfakes.FakeCacheInterface{}, &s3APIS, &digestionfakes.FakeDigestionBackend{}, &viewcountsfakes.FakeViewcountsCache{}))
	defer server.Close()

	client := lvs.NewLiveVideoServiceProtobufClient(server.URL, http.DefaultClient)
	header := make(http.Header)
	header.Set(auth.ForceAuthHeader, svc.customerID)
	ctx, err := twirp.WithHTTPRequestHeaders(context.Background(), header)
	require.NoError(t, err)

	_, err = client.CreateStreamKey(ctx, &lvs.CreateStreamKeyRequest{
		ContentId: "test",
		S3Bucket:  "test_bucket",
	})

	require.Error(t, err)
	require.True(t, strings.Contains(err.Error(), "please provide a bucket in us-west-2 region"))

}

func TestCreateStreamKeyCheckRegionError(t *testing.T) {
	svc, err := mockService()
	require.NoError(t, err)

	s3APIS := awsutilsfakes.FakeS3APIs{}
	s3APIS.GetBucketRegionReturns("", fmt.Errorf("%s %s", "NotFound", "Not Found"))
	server := httptest.NewServer(svc.service(&cachingfakes.FakeCacheInterface{}, &s3APIS, &digestionfakes.FakeDigestionBackend{}, &viewcountsfakes.FakeViewcountsCache{}))
	defer server.Close()

	client := lvs.NewLiveVideoServiceProtobufClient(server.URL, http.DefaultClient)
	header := make(http.Header)
	header.Set(auth.ForceAuthHeader, svc.customerID)
	ctx, err := twirp.WithHTTPRequestHeaders(context.Background(), header)
	require.NoError(t, err)

	_, err = client.CreateStreamKey(ctx, &lvs.CreateStreamKeyRequest{
		ContentId: "test",
		S3Bucket:  "test_bucket",
	})

	require.Error(t, err)
	require.True(t, strings.Contains(err.Error(), "bucket not found"))

}

func TestCreateStreamKeyDefaultValues(t *testing.T) {
	svc, err := mockService()
	require.NoError(t, err)

	s3APIS := awsutilsfakes.FakeS3APIs{}
	s3APIS.GetBucketRegionReturns("us-west-2", nil)
	server := httptest.NewServer(svc.service(&cachingfakes.FakeCacheInterface{}, &s3APIS, &digestionfakes.FakeDigestionBackend{}, &viewcountsfakes.FakeViewcountsCache{}))
	defer server.Close()

	client := lvs.NewLiveVideoServiceProtobufClient(server.URL, http.DefaultClient)
	header := make(http.Header)
	header.Set(auth.ForceAuthHeader, svc.customerID)
	ctx, err := twirp.WithHTTPRequestHeaders(context.Background(), header)
	require.NoError(t, err)

	resp, err := client.CreateStreamKey(ctx, &lvs.CreateStreamKeyRequest{
		ContentId: "test",
	})

	require.NoError(t, err)
	require.NotNil(t, resp)
	require.Equal(t, resp.ContentId, "test")
	require.Equal(t, resp.CustomerId, "test-customer")
	require.Equal(t, resp.StreamkeyMetadata.SnsNotificationEndpoint, "Not Specified")
	require.Equal(t, resp.StreamkeyMetadata.S3Bucket, "Not Specified")
	require.Equal(t, resp.StreamkeyMetadata.S3Prefix, "Not Specified")
	require.Equal(t, resp.StreamkeyMetadata.EnableLowLatency, false)
	require.Equal(t, resp.StreamkeyMetadata.LatencyMode, "")

	//Test the created stream key expiration time

	expirationTime, err := ptypes.Timestamp(resp.StreamkeyMetadata.ExpirationTime)
	require.Nil(t, err)

	currentTime := time.Now()

	timeToExpire := expirationTime.Sub(currentTime).Seconds()

	// Make sure the expiration time is not more than 10 seconds and greater than 0
	require.True(t, timeToExpire > 0)
	require.True(t, timeToExpire <= defaultStreamkeyExpiry)
}

func TestCreateStreamKeyInvalidCloudFrontURl(t *testing.T) {
	svc, err := mockService()
	require.NoError(t, err)

	s3APIS := awsutilsfakes.FakeS3APIs{}
	s3APIS.GetBucketRegionReturns("us-west-2", nil)
	server := httptest.NewServer(svc.service(&cachingfakes.FakeCacheInterface{}, &s3APIS, &digestionfakes.FakeDigestionBackend{}, &viewcountsfakes.FakeViewcountsCache{}))
	defer server.Close()

	client := lvs.NewLiveVideoServiceProtobufClient(server.URL, http.DefaultClient)
	header := make(http.Header)
	header.Set(auth.ForceAuthHeader, svc.customerID)
	ctx, err := twirp.WithHTTPRequestHeaders(context.Background(), header)
	require.NoError(t, err)

	_, err = client.CreateStreamKey(ctx, &lvs.CreateStreamKeyRequest{
		ContentId: "test",
		CdnUrl:    "http.com",
	})

	require.Error(t, err)
	require.True(t, strings.Contains(err.Error(), "invalid"))

}

func TestCreateStreamKeyInvalidLatencyMode(t *testing.T) {
	svc, err := mockService()
	require.NoError(t, err)

	s3APIS := awsutilsfakes.FakeS3APIs{}
	s3APIS.GetBucketRegionReturns("us-west-2", nil)
	server := httptest.NewServer(svc.service(&cachingfakes.FakeCacheInterface{}, &s3APIS, &digestionfakes.FakeDigestionBackend{}, &viewcountsfakes.FakeViewcountsCache{}))
	defer server.Close()

	client := lvs.NewLiveVideoServiceProtobufClient(server.URL, http.DefaultClient)
	header := make(http.Header)
	header.Set(auth.ForceAuthHeader, svc.customerID)
	ctx, err := twirp.WithHTTPRequestHeaders(context.Background(), header)
	require.NoError(t, err)

	_, err = client.CreateStreamKey(ctx, &lvs.CreateStreamKeyRequest{
		ContentId:   "test",
		LatencyMode: "invalid",
	})

	require.Error(t, err)
	require.True(t, strings.Contains(err.Error(), "accepatable values are normal or low"))
}

func TestGetStream(t *testing.T) {
	svc, err := mockService()
	require.NoError(t, err)
	cache := cachingfakes.FakeCacheInterface{}
	cacheGetStreamResponse := usher.UsherStreamResponse{
		Channel:      "lvs.test-customer.test",
		StartedOn:    time.Now().Unix(),
		ChannelCount: 0,
		Status:       "active",
		LVSMetadata:  `{"s3_vod_url":"test_url" , "sns_endpoint":"test_endpoint", "vod_manifest_url":"http://test.cloudfront.net/vod.m3u8"}`,
		CustomerId:   "test-customer",
		ContentId:    "test",
	}
	viewcountsBackend := viewcountsfakes.FakeViewcountsCache{}
	viewcountsBackend.GetCountReturns(uint64(5), nil)
	cache.GetStreamReturns(&cacheGetStreamResponse, nil)
	server := httptest.NewServer(svc.service(&cache, &awsutilsfakes.FakeS3APIs{}, &digestionfakes.FakeDigestionBackend{}, &viewcountsBackend))
	defer server.Close()

	client := lvs.NewLiveVideoServiceProtobufClient(server.URL, http.DefaultClient)
	header := make(http.Header)
	header.Set(auth.ForceAuthHeader, svc.customerID)
	ctx, err := twirp.WithHTTPRequestHeaders(context.Background(), header)
	require.NoError(t, err)

	resp, err := client.GetStream(ctx, &lvs.GetStreamRequest{
		ContentId: "test",
	})

	require.NoError(t, err)
	require.NotNil(t, resp)
	require.Equal(t, "live", resp.Stream.Status)
	require.Equal(t, "test_url", resp.Stream.S3VodUrl)
	require.Equal(t, resp.Stream.CdnPlaybackUrl,
		"https://usher.ttvnw.net/api/lvs/hls/lvs.test-customer.test.m3u8?allow_source=true&player_backend=mediaplayer")
	require.Equal(t, int64(5), resp.Stream.ViewerCount.Value)
	require.Equal(t, "http://test.cloudfront.net/vod.m3u8", resp.Stream.VodManifestUrl)
}

func TestGetStreamCacheError(t *testing.T) {
	svc, err := mockService()
	require.NoError(t, err)

	server := httptest.NewServer(svc.service(&cachingfakes.FakeCacheInterface{}, &awsutilsfakes.FakeS3APIs{}, &digestionfakes.FakeDigestionBackend{}, &viewcountsfakes.FakeViewcountsCache{}))
	defer server.Close()

	client := lvs.NewLiveVideoServiceProtobufClient(server.URL, http.DefaultClient)
	header := make(http.Header)
	header.Set(auth.ForceAuthHeader, svc.customerID)
	ctx, err := twirp.WithHTTPRequestHeaders(context.Background(), header)
	require.NoError(t, err)

	_, err = client.GetStream(ctx, &lvs.GetStreamRequest{
		ContentId: "test",
	})

	require.Error(t, err)
}

func TestGetStreamCacheNotFound(t *testing.T) {
	svc, err := mockService()
	require.NoError(t, err)

	cache := cachingfakes.FakeCacheInterface{}
	cache.GetStreamReturns(nil, twirp.NotFoundError("test-error"))
	server := httptest.NewServer(svc.service(&cache, &awsutilsfakes.FakeS3APIs{}, &digestionfakes.FakeDigestionBackend{}, &viewcountsfakes.FakeViewcountsCache{}))
	defer server.Close()

	client := lvs.NewLiveVideoServiceProtobufClient(server.URL, http.DefaultClient)
	header := make(http.Header)
	header.Set(auth.ForceAuthHeader, svc.customerID)
	ctx, err := twirp.WithHTTPRequestHeaders(context.Background(), header)
	require.NoError(t, err)

	_, err = client.GetStream(ctx, &lvs.GetStreamRequest{
		ContentId: "test",
	})
	require.True(t, strings.Contains(err.Error(), "not_found"))
}

func TestServiceListStreamsMissingArgument(t *testing.T) {
	svc, err := mockService()
	require.NoError(t, err)

	server := httptest.NewServer(svc.service(&cachingfakes.FakeCacheInterface{}, &awsutilsfakes.FakeS3APIs{}, &digestionfakes.FakeDigestionBackend{}, &viewcountsfakes.FakeViewcountsCache{}))
	defer server.Close()

	client := lvs.NewLiveVideoServiceProtobufClient(server.URL, http.DefaultClient)
	header := make(http.Header)
	header.Set(auth.ForceAuthHeader, svc.customerID)
	ctx, err := twirp.WithHTTPRequestHeaders(context.Background(), header)
	require.NoError(t, err)

	_, err = client.ListStreams(ctx, &lvs.ListStreamsRequest{})
	require.NotNil(t, err)
	require.True(t, strings.Contains(err.Error(), "invalid_argument"))
}

func TestServiceListStreamsInvalidArgumentType(t *testing.T) {
	svc, err := mockService()
	require.NoError(t, err)

	server := httptest.NewServer(svc.service(&cachingfakes.FakeCacheInterface{}, &awsutilsfakes.FakeS3APIs{}, &digestionfakes.FakeDigestionBackend{}, &viewcountsfakes.FakeViewcountsCache{}))
	defer server.Close()

	client := lvs.NewLiveVideoServiceProtobufClient(server.URL, http.DefaultClient)
	header := make(http.Header)
	header.Set(auth.ForceAuthHeader, svc.customerID)
	ctx, err := twirp.WithHTTPRequestHeaders(context.Background(), header)
	require.NoError(t, err)

	_, err = client.ListStreams(ctx, &lvs.ListStreamsRequest{
		Type: "invalid_type",
	})
	require.NotNil(t, err)
	require.True(t, strings.Contains(err.Error(), "invalid_argument"))
}

func TestServiceListStreams(t *testing.T) {
	svc, err := mockService()
	require.NoError(t, err)

	cacheGetStreamResponse1 := usher.UsherStreamResponse{
		Channel:      "lvs.test-customer.test",
		StartedOn:    time.Now().Unix(),
		ChannelCount: 1,
		Status:       "active",
		LVSMetadata:  `{"s3_vod_url":"test_url" , "sns_endpoint":"test_endpoint"}`,
	}

	cacheGetStreamResponse2 := usher.UsherStreamResponse{
		Channel:      "lvs.test-customer.test2",
		StartedOn:    time.Now().Unix(),
		ChannelCount: 2,
		Status:       "active",
	}

	var response []usher.UsherStreamResponse
	response = append(response, cacheGetStreamResponse1)
	response = append(response, cacheGetStreamResponse2)
	cache := cachingfakes.FakeCacheInterface{}
	cache.ListStreamsReturns(response, nil)

	viewcountsBackend := viewcountsfakes.FakeViewcountsCache{}
	viewcountsBackend.GetCountReturns(uint64(5), nil)

	server := httptest.NewServer(svc.service(&cache, &awsutilsfakes.FakeS3APIs{}, &digestionfakes.FakeDigestionBackend{}, &viewcountsBackend))
	defer server.Close()

	client := lvs.NewLiveVideoServiceProtobufClient(server.URL, http.DefaultClient)
	header := make(http.Header)
	header.Set(auth.ForceAuthHeader, svc.customerID)
	ctx, err := twirp.WithHTTPRequestHeaders(context.Background(), header)
	require.NoError(t, err)

	resp, err := client.ListStreams(ctx, &lvs.ListStreamsRequest{
		Type: "all",
	})
	require.Equal(t, 2, len(resp.Streams))
	require.NoError(t, err)
	require.Equal(t, "test_url", resp.Streams[0].S3VodUrl)
	val := &wrappers.Int64Value{Value: int64(5)}
	require.Equal(t, val, resp.Streams[0].ViewerCount)
	require.Equal(t, val, resp.Streams[1].ViewerCount)

}

func TestServiceListStreamsCacheError(t *testing.T) {
	svc, err := mockService()
	require.NoError(t, err)
	cache := cachingfakes.FakeCacheInterface{}
	cache.ListStreamsReturns(nil, fmt.Errorf("test error"))

	server := httptest.NewServer(svc.service(&cache, &awsutilsfakes.FakeS3APIs{}, &digestionfakes.FakeDigestionBackend{}, &viewcountsfakes.FakeViewcountsCache{}))

	defer server.Close()

	client := lvs.NewLiveVideoServiceProtobufClient(server.URL, http.DefaultClient)
	header := make(http.Header)
	header.Set(auth.ForceAuthHeader, svc.customerID)
	ctx, err := twirp.WithHTTPRequestHeaders(context.Background(), header)
	require.NoError(t, err)

	_, err = client.ListStreams(ctx, &lvs.ListStreamsRequest{
		Type: "all",
	})
	require.Error(t, err)
}

func TestServiceListStreamsCacheNotFound(t *testing.T) {
	svc, err := mockService()
	require.NoError(t, err)

	cache := cachingfakes.FakeCacheInterface{}
	cache.ListStreamsReturns(nil, twirp.NotFoundError("test-error"))

	server := httptest.NewServer(svc.service(&cache, &awsutilsfakes.FakeS3APIs{}, &digestionfakes.FakeDigestionBackend{}, &viewcountsfakes.FakeViewcountsCache{}))
	defer server.Close()

	client := lvs.NewLiveVideoServiceProtobufClient(server.URL, http.DefaultClient)
	header := make(http.Header)
	header.Set(auth.ForceAuthHeader, svc.customerID)
	ctx, err := twirp.WithHTTPRequestHeaders(context.Background(), header)
	require.NoError(t, err)

	_, err = client.ListStreams(ctx, &lvs.ListStreamsRequest{
		Type: "all",
	})
	require.True(t, strings.Contains(err.Error(), "not_found"))
}

func TestServiceListStreamsFiltering(t *testing.T) {
	svc, err := mockService()
	require.NoError(t, err)

	cache := cachingfakes.FakeCacheInterface{}
	cacheGetStreamResponse1 := usher.UsherStreamResponse{
		Channel:      "lvs.test-customer.test",
		StartedOn:    time.Now().Unix(),
		ChannelCount: 0,
		Status:       "pending",
		LVSMetadata:  `{"s3_vod_url":"test_url" , "sns_endpoint":"test_endpoint"}`,
	}

	cacheGetStreamResponse2 := usher.UsherStreamResponse{
		Channel:      "lvs.test-customer.test2",
		StartedOn:    time.Now().Unix(),
		ChannelCount: 0,
		Status:       "active",
	}

	var response []usher.UsherStreamResponse
	response = append(response, cacheGetStreamResponse1)
	response = append(response, cacheGetStreamResponse2)

	cache.ListStreamsReturns(response, nil)

	server := httptest.NewServer(svc.service(&cache, &awsutilsfakes.FakeS3APIs{}, &digestionfakes.FakeDigestionBackend{}, &viewcountsfakes.FakeViewcountsCache{}))
	defer server.Close()

	client := lvs.NewLiveVideoServiceProtobufClient(server.URL, http.DefaultClient)
	header := make(http.Header)
	header.Set(auth.ForceAuthHeader, svc.customerID)
	ctx, err := twirp.WithHTTPRequestHeaders(context.Background(), header)
	require.NoError(t, err)

	// Request all streams , nothing should be filtered
	resp, err := client.ListStreams(ctx, &lvs.ListStreamsRequest{
		Type: "all",
	})
	require.Equal(t, 2, len(resp.Streams))
	require.NoError(t, err)
	require.Equal(t, "test_url", resp.Streams[0].S3VodUrl)

	// Request only live streams , nothing should be filtered
	resp, err = client.ListStreams(ctx, &lvs.ListStreamsRequest{
		Type: "live",
	})
	require.Equal(t, 1, len(resp.Streams))
	require.NoError(t, err)
	require.Equal(t, "live", resp.Streams[0].Status)

	// Request only pending streams , nothing should be filtered
	resp, err = client.ListStreams(ctx, &lvs.ListStreamsRequest{
		Type: "preparing",
	})
	require.Equal(t, 1, len(resp.Streams))
	require.NoError(t, err)
	require.Equal(t, "preparing", resp.Streams[0].Status)

}

func TestServiceDecodeStreamKey(t *testing.T) {
	svc, err := mockService()
	require.NoError(t, err)
	s3APIs := awsutilsfakes.FakeS3APIs{}
	s3APIs.GetBucketRegionReturns("us-west-2", nil)
	server := httptest.NewServer(svc.service(&cachingfakes.FakeCacheInterface{}, &s3APIs, &digestionfakes.FakeDigestionBackend{}, &viewcountsfakes.FakeViewcountsCache{}))
	defer server.Close()

	client := lvs.NewLiveVideoServiceProtobufClient(server.URL, http.DefaultClient)
	header := make(http.Header)
	header.Set(auth.ForceAuthHeader, svc.customerID)
	ctx, err := twirp.WithHTTPRequestHeaders(context.Background(), header)
	require.NoError(t, err)

	resp, err := client.CreateStreamKey(ctx, &lvs.CreateStreamKeyRequest{
		ContentId:               "test",
		S3Bucket:                "testbucket",
		S3Prefix:                "testprefix",
		SnsNotificationEndpoint: "arn:aws:sns:us-west-2:123456:test",
		TtlSeconds:              10,
		EnableLowLatency:        true,
		CdnUrl:                  "http://test.cloudfront.net/",
		LatencyMode:             "low",
	})

	require.Nil(t, err)
	require.NotNil(t, resp)

	//Decode the streamkey and make sure we get same data back

	decodedResponse, err := client.DecodeStreamKey(ctx, &lvs.DecodeStreamKeyRequest{
		Streamkey: resp.GetStreamkey(),
	})

	require.NotNil(t, decodedResponse)
	require.Nil(t, err)
	require.Equal(t, "test-customer", decodedResponse.GetCustomerId())
	require.Equal(t, "test", decodedResponse.GetContentId())
	require.Equal(t, "testbucket", decodedResponse.GetStreamkeyMetadata().GetS3Bucket())
	require.Equal(t, "testprefix", decodedResponse.GetStreamkeyMetadata().GetS3Prefix())
	require.Equal(t, "arn:aws:sns:us-west-2:123456:test", decodedResponse.GetStreamkeyMetadata().GetSnsNotificationEndpoint())
	require.Equal(t, true, decodedResponse.GetStreamkeyMetadata().GetEnableLowLatency())
	require.Equal(t, "http://test.cloudfront.net/", decodedResponse.GetStreamkeyMetadata().GetCdnUrl())
	require.Equal(t, "low", decodedResponse.GetStreamkeyMetadata().GetLatencyMode())

}

func TestDecodeStreamKeyError(t *testing.T) {
	svc, err := mockService()
	require.NoError(t, err)
	server := httptest.NewServer(svc.service(&cachingfakes.FakeCacheInterface{}, &awsutilsfakes.FakeS3APIs{}, &digestionfakes.FakeDigestionBackend{}, &viewcountsfakes.FakeViewcountsCache{}))
	defer server.Close()

	client := lvs.NewLiveVideoServiceProtobufClient(server.URL, http.DefaultClient)
	header := make(http.Header)
	header.Set(auth.ForceAuthHeader, svc.customerID)
	ctx, err := twirp.WithHTTPRequestHeaders(context.Background(), header)
	require.NoError(t, err)

	//Decode the streamkey and make sure we get same data back

	_, err = client.DecodeStreamKey(ctx, &lvs.DecodeStreamKeyRequest{
		Streamkey: "lvs.notavalidpublic.notvalidprivatestreamkeydata",
	})

	require.NotNil(t, err)
	require.Equal(t, twirp.NewError(twirp.InvalidArgument, "streamkey is invalid"), err)

}

func TestUnauthorizedStremKeyDecryption(t *testing.T) {
	svc, err := mockService()
	require.NoError(t, err)

	s3APIs := awsutilsfakes.FakeS3APIs{}
	s3APIs.GetBucketRegionReturns("us-west-2", nil)
	server := httptest.NewServer(svc.service(&cachingfakes.FakeCacheInterface{}, &s3APIs, &digestionfakes.FakeDigestionBackend{}, &viewcountsfakes.FakeViewcountsCache{}))
	defer server.Close()

	client := lvs.NewLiveVideoServiceProtobufClient(server.URL, http.DefaultClient)
	header := make(http.Header)
	header.Set(auth.ForceAuthHeader, svc.customerID)
	ctx, err := twirp.WithHTTPRequestHeaders(context.Background(), header)
	require.NoError(t, err)

	//Get public portion of lvsapi with a customer id not as same as certificate

	ts, err := ptypes.TimestampProto(time.Now().Add(10))
	require.Nil(t, err)
	sk := streamkey.NewV1("invalid_customerid", &streamkey.PrivateData{ExpirationTime: ts})
	pubMsg, err := proto.Marshal(sk.Pub)
	require.Nil(t, err)
	pubMsgStr := base64.RawURLEncoding.EncodeToString(pubMsg)

	//Decode the streamkey and make sure we get same data back

	_, err = client.DecodeStreamKey(ctx, &lvs.DecodeStreamKeyRequest{
		Streamkey: fmt.Sprintf("%s.%s.%s", "lvs", pubMsgStr, "invalidprivatedata"),
	})

	require.NotNil(t, err)
	require.Equal(t, twirp.NewError(twirp.PermissionDenied, "Not Authorized to decrypt streamkey"), err)

}

func TestStopStreamSuccess(t *testing.T) {
	svc, err := mockService()
	require.NoError(t, err)
	cache := cachingfakes.FakeCacheInterface{}
	digestion := digestionfakes.FakeDigestionBackend{}
	digestion.DeleteStreamReturns(nil)

	server := httptest.NewServer(svc.service(&cache, &awsutilsfakes.FakeS3APIs{}, &digestion, &viewcountsfakes.FakeViewcountsCache{}))
	defer server.Close()

	client := lvs.NewLiveVideoServiceProtobufClient(server.URL, http.DefaultClient)
	header := make(http.Header)
	header.Set(auth.ForceAuthHeader, svc.customerID)
	ctx, err := twirp.WithHTTPRequestHeaders(context.Background(), header)
	require.NoError(t, err)

	resp, err := client.StopStream(ctx, &lvs.StopStreamRequest{
		ContentId: "test",
	})

	require.NoError(t, err)
	require.NotNil(t, resp)

}

func TestStopStreamNotFound(t *testing.T) {
	svc, err := mockService()
	require.NoError(t, err)
	cache := cachingfakes.FakeCacheInterface{}
	digestion := digestionfakes.FakeDigestionBackend{}
	digestion.DeleteStreamReturns(fmt.Errorf("404 not found"))

	server := httptest.NewServer(svc.service(&cache, &awsutilsfakes.FakeS3APIs{}, &digestion, &viewcountsfakes.FakeViewcountsCache{}))
	defer server.Close()

	client := lvs.NewLiveVideoServiceProtobufClient(server.URL, http.DefaultClient)
	header := make(http.Header)
	header.Set(auth.ForceAuthHeader, svc.customerID)
	ctx, err := twirp.WithHTTPRequestHeaders(context.Background(), header)
	require.NoError(t, err)

	_, err = client.StopStream(ctx, &lvs.StopStreamRequest{
		ContentId: "test",
	})

	require.Error(t, err)
	require.True(t, strings.Contains(err.Error(), "not_found"))

}

func TestStopStreamOtherError(t *testing.T) {
	svc, err := mockService()
	require.NoError(t, err)
	cache := cachingfakes.FakeCacheInterface{}
	digestion := digestionfakes.FakeDigestionBackend{}
	digestion.DeleteStreamReturns(fmt.Errorf("500 internal"))

	server := httptest.NewServer(svc.service(&cache, &awsutilsfakes.FakeS3APIs{}, &digestion, &viewcountsfakes.FakeViewcountsCache{}))
	defer server.Close()

	client := lvs.NewLiveVideoServiceProtobufClient(server.URL, http.DefaultClient)
	header := make(http.Header)
	header.Set(auth.ForceAuthHeader, svc.customerID)
	ctx, err := twirp.WithHTTPRequestHeaders(context.Background(), header)
	require.NoError(t, err)

	_, err = client.StopStream(ctx, &lvs.StopStreamRequest{
		ContentId: "test",
	})

	require.Error(t, err)
	fmt.Println(err.Error())
	require.True(t, strings.Contains(err.Error(), "twirp error internal"))

}
