package usher

import (
	"context"
	"fmt"
	"strings"
	"time"

	"code.justin.tv/video/usherapi/rpc/usher"
	"github.com/cactus/go-statsd-client/statsd"
)

// Client defines Twirp RPC calls to fetch transcode data.
type Client interface {
	HLSTranscodeLVSCustomerTranscodes(context.Context, *usher.HLSTranscodeLVSCustomerTranscodesRequest) (*usher.ListOf_HLSTranscode, error)
	HLSTranscodeLVSChannel(context.Context, *usher.HLSTranscodeLVSChannelRequest) (*usher.HLSTranscode, error)
}

// UsherBackend is an Interface which defines the Usher API's used to fetch data from Usher
type UsherBackend interface {
	ListStreams(customerId string) ([]UsherStreamResponse, error)
	GetStreamInfo(customerId string, contentId string) (*UsherStreamResponse, error)
}

// Usher is just a struct which ties up the API's and make testing easier
type Usher struct {
	clients []Client
	statter statsd.Statter
}

// NewUsher function returns instance of Usher with the client set
func NewUsher(clients []Client, s statsd.Statter) *Usher {
	usher := Usher{
		clients: clients,
		statter: s,
	}
	return &usher
}

// UsherStreamResponse holds the basic usher data to populate an LVS response
type UsherStreamResponse struct {
	Channel      string `json:"channel"`
	StartedOn    int64  `json:"started_on"`
	ChannelCount int    `json:"channel_count"`
	LVSMetadata  string `json:"lvs_metadata"`
	Status       string `json:"status"`
	CustomerId   string `json:"customer_id"`
	ContentId    string `json:"content_id"`
	StreamId     uint64 `json:"destination"`
}

// ListStreams will return a list of usher streams for given customerId
func (u *Usher) ListStreams(customerId string) ([]UsherStreamResponse, error) {
	var us []UsherStreamResponse
	var errors []string
	for _, client := range u.clients {
		start := time.Now()
		resp, err := client.HLSTranscodeLVSCustomerTranscodes(context.Background(),
			&usher.HLSTranscodeLVSCustomerTranscodesRequest{
				CustomerId: customerId,
			})
		if u.statter != nil {
			_ = u.statter.TimingDuration("usher.list_streams", time.Since(start), 1.0)
		}
		if err != nil {
			errors = append(errors, err.Error())
			continue
		}
		for _, t := range resp.Transcodes {
			us = append(us, convertProtoTranscode(t))
		}
	}
	if len(us) == 0 && len(errors) > 0 {
		return us, fmt.Errorf("Get lvs customer transcodes from usher failed: %s", strings.Join(errors, ", "))
	}
	return us, nil
}

// GetStreamInfo return the information around a given usher stream in its streams table
func (u *Usher) GetStreamInfo(customerId string, contentId string) (*UsherStreamResponse, error) {
	var errors []string
	for _, client := range u.clients {
		start := time.Now()
		resp, err := client.HLSTranscodeLVSChannel(context.Background(),
			&usher.HLSTranscodeLVSChannelRequest{
				CustomerId: customerId,
				ContentId:  contentId,
			})
		if u.statter != nil {
			_ = u.statter.TimingDuration("usher.get_stream_info", time.Since(start), 1.0)
		}
		if err != nil {
			errors = append(errors, err.Error())
			continue
		}
		us := convertProtoTranscode(resp)
		return &us, nil
	}
	return nil, fmt.Errorf("Get lvs channel from usher failed: %s", strings.Join(errors, ", "))
}

func convertProtoTranscode(t *usher.HLSTranscode) UsherStreamResponse {
	return UsherStreamResponse{
		ChannelCount: int(t.ChannelCount),
		Channel:      t.Channel,
		StartedOn:    t.StartTimeSeconds,
		LVSMetadata:  t.LvsMetadataRawString,
		Status:       t.Status,
		CustomerId:   t.CustomerId,
		ContentId:    t.ContentId,
		StreamId:     uint64(t.StreamId),
	}
}
