package usher_test

import (
	"errors"
	"testing"

	usherpb "code.justin.tv/video/usherapi/rpc/usher"

	"code.justin.tv/video/lvsapi/internal/usher"
	"code.justin.tv/video/lvsapi/internal/usher/usherfakes"
	"github.com/stretchr/testify/require"
)

func TestGetStreamInfo(t *testing.T) {
	t.Run("one successful response", func(t *testing.T) {
		mockClient1 := &usherfakes.FakeClient{}
		mockClient1.HLSTranscodeLVSChannelReturns(&usherpb.HLSTranscode{
			CustomerId: "amzn_live",
			ContentId:  "test",
			Status:     "active",
		}, nil)
		mockClient2 := &usherfakes.FakeClient{}
		mockClient2.HLSTranscodeLVSChannelReturns(nil, errors.New("twirp error"))

		u := usher.NewUsher([]usher.Client{mockClient1, mockClient2}, nil)
		response, err := u.GetStreamInfo("amzn_live", "test")
		require.NotNil(t, response)
		require.Nil(t, err)
		require.Equal(t, "amzn_live", response.CustomerId)
		require.Equal(t, "test", response.ContentId)
		require.Equal(t, "active", response.Status)
		require.Equal(t, int64(0), response.StartedOn)
	})

	t.Run("two successful responses", func(t *testing.T) {
		mockClient1 := &usherfakes.FakeClient{}
		mockClient1.HLSTranscodeLVSChannelReturns(&usherpb.HLSTranscode{
			CustomerId: "amzn_live",
			ContentId:  "test",
			Status:     "active",
		}, nil)
		mockClient2 := &usherfakes.FakeClient{}
		mockClient2.HLSTranscodeLVSChannelReturns(&usherpb.HLSTranscode{
			CustomerId: "amzn_live",
			ContentId:  "test2",
			Status:     "active",
		}, nil)

		u := usher.NewUsher([]usher.Client{mockClient1, mockClient2}, nil)
		response, err := u.GetStreamInfo("amzn_live", "test")
		require.NotNil(t, response)
		require.Nil(t, err)
		require.Equal(t, "amzn_live", response.CustomerId)
		require.Equal(t, "test", response.ContentId)
		require.Equal(t, "active", response.Status)
		require.Equal(t, int64(0), response.StartedOn)
	})

	t.Run("all twirp errors", func(t *testing.T) {
		mockClient1 := &usherfakes.FakeClient{}
		mockClient1.HLSTranscodeLVSChannelReturns(nil, errors.New("twirp error"))
		mockClient2 := &usherfakes.FakeClient{}
		mockClient2.HLSTranscodeLVSChannelReturns(nil, errors.New("twirp error"))

		u := usher.NewUsher([]usher.Client{mockClient1, mockClient2}, nil)
		_, err := u.GetStreamInfo("amzn_live", "test")
		require.NotNil(t, err)
	})
}

func TestListStreams(t *testing.T) {
	t.Run("one successful response", func(t *testing.T) {
		mockClient1 := &usherfakes.FakeClient{}
		mockClient1.HLSTranscodeLVSCustomerTranscodesReturns(&usherpb.ListOf_HLSTranscode{
			Transcodes: []*usherpb.HLSTranscode{
				&usherpb.HLSTranscode{
					CustomerId:   "amzn_live",
					ContentId:    "test",
					Status:       "active",
					ChannelCount: 2,
				},
				&usherpb.HLSTranscode{
					CustomerId:   "amzn_live",
					ContentId:    "test2",
					Status:       "pending",
					ChannelCount: 1,
				},
			},
		}, nil)
		mockClient2 := &usherfakes.FakeClient{}
		mockClient2.HLSTranscodeLVSCustomerTranscodesReturns(nil, errors.New("twirp error"))

		u := usher.NewUsher([]usher.Client{mockClient1, mockClient2}, nil)
		response, err := u.ListStreams("amzn_live")
		require.NotNil(t, response)
		require.Nil(t, err)
		require.Equal(t, response, []usher.UsherStreamResponse{
			usher.UsherStreamResponse{CustomerId: "amzn_live", ContentId: "test", Status: "active", ChannelCount: 2},
			usher.UsherStreamResponse{CustomerId: "amzn_live", ContentId: "test2", Status: "pending", ChannelCount: 1},
		})
	})

	t.Run("two successful responses", func(t *testing.T) {
		mockClient1 := &usherfakes.FakeClient{}
		mockClient1.HLSTranscodeLVSCustomerTranscodesReturns(&usherpb.ListOf_HLSTranscode{
			Transcodes: []*usherpb.HLSTranscode{
				&usherpb.HLSTranscode{
					CustomerId:   "amzn_live",
					ContentId:    "test",
					Status:       "active",
					ChannelCount: 2,
				},
				&usherpb.HLSTranscode{
					CustomerId:   "amzn_live",
					ContentId:    "test2",
					Status:       "pending",
					ChannelCount: 1,
				},
			},
		}, nil)
		mockClient2 := &usherfakes.FakeClient{}
		mockClient2.HLSTranscodeLVSCustomerTranscodesReturns(&usherpb.ListOf_HLSTranscode{
			Transcodes: []*usherpb.HLSTranscode{
				&usherpb.HLSTranscode{
					CustomerId:   "amzn_live",
					ContentId:    "test3",
					Status:       "active",
					ChannelCount: 2,
				},
				&usherpb.HLSTranscode{
					CustomerId:   "amzn_live",
					ContentId:    "test4",
					Status:       "pending",
					ChannelCount: 1,
				},
			},
		}, nil)

		u := usher.NewUsher([]usher.Client{mockClient1, mockClient2}, nil)
		response, err := u.ListStreams("amzn_live")
		require.NotNil(t, response)
		require.Nil(t, err)
		require.Equal(t, response, []usher.UsherStreamResponse{
			usher.UsherStreamResponse{CustomerId: "amzn_live", ContentId: "test", Status: "active", ChannelCount: 2},
			usher.UsherStreamResponse{CustomerId: "amzn_live", ContentId: "test2", Status: "pending", ChannelCount: 1},
			usher.UsherStreamResponse{CustomerId: "amzn_live", ContentId: "test3", Status: "active", ChannelCount: 2},
			usher.UsherStreamResponse{CustomerId: "amzn_live", ContentId: "test4", Status: "pending", ChannelCount: 1},
		})
	})

	t.Run("all twirp errors", func(t *testing.T) {
		mockClient1 := &usherfakes.FakeClient{}
		mockClient1.HLSTranscodeLVSCustomerTranscodesReturns(nil, errors.New("twirp error"))
		mockClient2 := &usherfakes.FakeClient{}
		mockClient2.HLSTranscodeLVSCustomerTranscodesReturns(nil, errors.New("twirp error"))

		u := usher.NewUsher([]usher.Client{mockClient1, mockClient2}, nil)
		_, err := u.ListStreams("amzn_live")
		require.NotNil(t, err)
	})
}
