package utils

import (
	"flag"
	"os"
	"testing"
	"time"

	"github.com/stretchr/testify/require"
)

// define some test flags
var (
	testKey1 = flag.String("test-key-1", "", "test key 1")
	testKey2 = flag.String("test-key-2", "", "test key 2")
	testKey3 = flag.Duration("test-key-3", 60*time.Second, "test key 3")
)

var envMap = map[string]string{
	"test-key-1": "TEST_KEY_1",
	"test-key-2": "TEST_KEY_2",
	"test-key-3": "TEST_KEY_3",
}

func setupEnvVariables(t *testing.T, vars map[string]string) func(t *testing.T) {
	for key, val := range vars {
		require.NoError(t, os.Setenv(key, val))
	}

	return func(t *testing.T) {
		for key := range vars {
			require.NoError(t, os.Unsetenv(key))
		}
	}
}

func resetFlags() {
	flag.Set("test-key-1", "")
	flag.Set("test-key-2", "")
	flag.Set("test-key-3", "1m0s")
}

func TestValidateFlags(t *testing.T) {
	defer resetFlags()

	flag.Set("test-key-1", "test-1")
	flag.Set("test-key-2", "test-2")
	missing, err := ValidateFlags(envMap)
	require.Empty(t, missing)
	require.Nil(t, err)
}

func TestValidateFlagsMissing(t *testing.T) {
	defer resetFlags()

	flag.Set("test-key-1", "test-1")
	missing, err := ValidateFlags(envMap)
	require.Contains(t, missing, "TEST_KEY_2")
	require.Nil(t, err)
}

func TestValidateFlagsEnvOverride(t *testing.T) {
	defer resetFlags()

	envVars := make(map[string]string)
	envVars["TEST_KEY_1"] = "test-1"
	envVars["TEST_KEY_2"] = "test-2"
	teardownEnvVariables := setupEnvVariables(t, envVars)
	defer teardownEnvVariables(t)

	missing, err := ValidateFlags(envMap)
	require.Empty(t, missing)
	require.Nil(t, err)
}

func TestValidateFlagsEnvOverrideMissing(t *testing.T) {
	defer resetFlags()

	envVars := make(map[string]string)
	envVars["TEST_KEY_1"] = "test-1"
	teardownEnvVariables := setupEnvVariables(t, envVars)
	defer teardownEnvVariables(t)

	missing, err := ValidateFlags(envMap)
	require.Contains(t, missing, "TEST_KEY_2")
	require.Nil(t, err)
}

func TestValidateFlagsEnvOverrideFailure(t *testing.T) {
	defer resetFlags()

	envVars := make(map[string]string)
	envVars["TEST_KEY_1"] = "test-1"
	envVars["TEST_KEY_2"] = "test-1"
	envVars["TEST_KEY_3"] = "invalid-val-for-duration"
	teardownEnvVariables := setupEnvVariables(t, envVars)
	defer teardownEnvVariables(t)

	_, err := ValidateFlags(envMap)
	require.NotNil(t, err)
}
