package viewcounts

import (
	"context"
	"fmt"
	"net/http"
	"time"

	"code.justin.tv/video/lvsapi/internal/constants"
	"code.justin.tv/video/lvsapi/internal/logging"
	"code.justin.tv/video/viewcount-api/api/viewcounts"

	"github.com/cactus/go-statsd-client/statsd"
	"github.com/twitchtv/twirp"
)

//ViewcountsClient holds the viewcounts api client
type ViewcountsClient struct {
	client  viewcounts.Viewcounts
	statter statsd.Statter
}

const viewcountsTimeout = time.Second * 2

//NewViewcountsClient returns a new viewcount api clinet to be used
func NewViewcountsClient(viewcountsBaseUrl string, s statsd.Statter) *ViewcountsClient {
	hc := &http.Client{
		Timeout: viewcountsTimeout,
	}

	return &ViewcountsClient{
		client:  viewcounts.NewViewcountsJSONClient(viewcountsBaseUrl, hc),
		statter: s,
	}
}

//GetCountsForStream call viewcount api and returns the counts for a given streamID
func (vc *ViewcountsClient) GetCountsForStream(ctx context.Context, streamId uint64) (*viewcounts.Views, error) {
	if vc.client == nil {
		return nil, fmt.Errorf("Invalid client")
	}

	if streamId == 0 {
		return nil, fmt.Errorf("Invalid Stream ID")
	}

	streamIdRequest := viewcounts.StreamID{
		Id: streamId,
	}

	vc.incrementStat("viewcount_api_requests")
	views, err := vc.client.ForStream(ctx, &streamIdRequest)
	if err != nil {
		logging.Info(ctx, "viewcounts api for streamId:%d threw error: %v", streamId, err)
		if twirpErr, ok := err.(twirp.Error); ok {
			vc.incrementStat(fmt.Sprintf("%s.%s", constants.METRICS_HTTP_ERROR, twirpErr.Code()))
		}
		return nil, err
	}
	vc.incrementStat("viewcount_api_success")

	return views, nil
}

func (vc *ViewcountsClient) incrementStat(dimension string) {
	if vc.statter != nil {
		_ = vc.statter.Inc(dimension, 1, 1.0)
	}
}
